/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) NRAO - National Radio Astronomy Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.tmcdb.alarms.ui.widgets;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.EventChannel;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.editors.inputs.ChannelMappingEditorInput;
import alma.obops.tmcdbgui.handlers.NewChannelMappingAction;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;
import alma.obops.tmcdbgui.widgets.DirtyPublishingComposite;

public class ChannelMappingEventChannelNameComposite extends DirtyPublishingComposite 
{
	private static final String NOTICE = "Notice";
	private static final String NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP = "No selections made; dialog is only for testing regexp.";
	private Label channelLabel, regExpLabel;
	private Button regExpCheckBox, testButton;
	private Combo channelNameCombo, notificationServiceCombo;
	private Text regExpText;
	private Configuration configuration;
	
	public ChannelMappingEventChannelNameComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}
	
	@Override
	public boolean setFocus() {
		return notificationServiceCombo.setFocus();
	}

	private void initialize() {
		createComposite();
	}

	/**
	 * This method initializes composite	
	 *
	 */
	private void createComposite() 
	{	
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = false;
		setLayout(gridLayout);
		
		createNotificationServiceWidgets();	
		createChannelNameWidgets();	
		createRegExpWidgets();
		setDirty(false);
	}

	private void createRegExpWidgets() 
	{
		createRegExpLabelAndText();
		createRegExpCheckbox();
		createRegExpTestButton();
	}

	private void createRegExpTestButton() 
	{
		testButton = new Button(this, SWT.PUSH);
		GridData gd1 = new GridData();
		gd1.grabExcessHorizontalSpace = false;
		gd1.grabExcessVerticalSpace = false;
		gd1.horizontalAlignment = SWT.LEFT;
		gd1.verticalAlignment = SWT.CENTER;
		gd1.horizontalSpan = 1;
		testButton.setText("Test regexp");
		testButton.setLayoutData(gd1);
		testButton.addListener(SWT.Selection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				String regExp = regExpCheckBox.getSelection() ? regExpText.getText() : channelNameCombo.getItem(channelNameCombo.getSelectionIndex());
				try {
					EventChannel[] evtChannels = SwConfigurationConversationUtils.getInstance().findEventChannelsByRegEx(regExp, configuration);
					ElementListSelectionDialog d = new ElementListSelectionDialog(getShell(), new RegExpHitsLabelProvider());
					d.setTitle("Regexp results");
					d.setElements(evtChannels);
					d.setMultipleSelection(false);
					d.setInitialSelections(new String[0]);
					d.open();
					if(d.getReturnCode() == ElementListSelectionDialog.OK) {
						MessageDialog.openInformation(getShell(), NOTICE, NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP);
					}
				} catch (Exception e) {
					e.printStackTrace();
					MessageDialog.openError(getShell(), "Unexpected error", e.getMessage());
				}
			}
		});
	}

	private static class RegExpHitsLabelProvider implements ILabelProvider
	{
		@Override
		public Image getImage(Object arg0) {
			return RcpUtils.getImage("icons/nc.png");
		}

		@Override
		public String getText(Object obj) 
		{
			if(obj instanceof EventChannel) {
				EventChannel ec = (EventChannel) obj;
				return ec.getName();
			}
			return null;
		}

		@Override
		public void addListener(ILabelProviderListener arg0) {
			// TODO Auto-generated method stub
		}

		@Override
		public void dispose() {
			// TODO Auto-generated method stub
		}

		@Override
		public boolean isLabelProperty(Object arg0,
				String arg1) 
		{
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public void removeListener(
				ILabelProviderListener arg0) 
		{
			// TODO Auto-generated method stub
		}
	}
	
	
	private void createRegExpCheckbox() 
	{
		regExpCheckBox = new Button(this, SWT.CHECK);
		GridData gd2 = new GridData();
		gd2.grabExcessHorizontalSpace = false;
		gd2.grabExcessVerticalSpace = false;
		gd2.horizontalAlignment = SWT.LEFT;
		gd2.verticalAlignment = SWT.CENTER;
		gd2.horizontalSpan = 1;
		regExpCheckBox.setText("Use regexp");
		regExpCheckBox.setLayoutData(gd2);
		regExpCheckBox.setSelection(true);
		regExpCheckBox.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				if(regExpCheckBox.getSelection()) {
					testButton.setEnabled(true);
					regExpText.setEnabled(true);
					regExpLabel.setEnabled(true);
					channelLabel.setEnabled(false);
					channelNameCombo.setEnabled(false);
				} else {
					testButton.setEnabled(false);
					regExpText.setEnabled(false);
					regExpLabel.setEnabled(false);
					channelLabel.setEnabled(true);
					channelNameCombo.setEnabled(true);
				}
				setDirty(true);
			}
		});
	}

	private void createRegExpLabelAndText() 
	{
		regExpLabel = new Label(this, SWT.NONE);
		regExpLabel.setText("Regexp for event channel");
		GridData gd4 = new GridData();
		gd4.grabExcessHorizontalSpace = false;
		gd4.grabExcessVerticalSpace = false;
		gd4.horizontalAlignment = SWT.LEFT;
		gd4.verticalAlignment = SWT.CENTER;
		regExpLabel.setLayoutData(gd4);
		
		regExpText = new Text(this, SWT.BORDER);
		GridData gd5 = new GridData();
		gd5.grabExcessHorizontalSpace = true;
		gd5.grabExcessVerticalSpace = false;
		gd5.horizontalAlignment = SWT.FILL;
		gd5.verticalAlignment = SWT.CENTER;
		regExpText.setLayoutData(gd5);
		
		regExpText.addModifyListener(new ModifyListener() {

			@Override
			public void modifyText(ModifyEvent e) {
				setDirty(true);
				int index = channelNameCombo.indexOf(regExpText.getText());
				if(index == -1) {
					channelNameCombo.deselectAll();
				} else {
					channelNameCombo.select(index);	
				}
			}
			
		});
	}

	private void createChannelNameWidgets() 
	{
		channelLabel = new Label(this, SWT.NONE);
		channelLabel.setText("Select event channel");
		GridData gd6 = new GridData();
		gd6.grabExcessHorizontalSpace = false;
		gd6.grabExcessVerticalSpace = false;
		gd6.horizontalAlignment = SWT.LEFT;
		gd6.verticalAlignment = SWT.CENTER;
		channelLabel.setEnabled(false);
		channelLabel.setLayoutData(gd6);
		
		channelNameCombo = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gd3 = new GridData();
		gd3.grabExcessHorizontalSpace = true;
		gd3.grabExcessVerticalSpace = false;
		gd3.horizontalSpan = 1;
		gd3.horizontalAlignment = SWT.FILL;
		channelNameCombo.setEnabled(false);
		channelNameCombo.setLayoutData(gd3);
		
		channelNameCombo.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
				regExpText.setText(channelNameCombo.getItem(channelNameCombo.getSelectionIndex()));
			}
			
		});	
	}

	private String[] getEventChannelStrings() 
	{
		String [] retVal = null;
		
		try 
		{
			SwConfigurationConversationUtils.getInstance().hydrateEventChannels(configuration);
			if(null != configuration.getEventChannels()) {
				retVal = new String[configuration.getEventChannels().size()];
				int count = 0;
				for(EventChannel channel : configuration.getEventChannels()) {
					retVal[count++] = channel.getName();
				}
			}
		} 
		catch (Exception e) 
		{
			throw new RuntimeException("Could not get names of event channels");
		}
		
		return retVal;
	}

	private void createNotificationServiceWidgets() 
	{
		Label label3 = new Label(this, SWT.NONE);
		label3.setText("Select notification service");
		GridData gd7 = new GridData();
		gd7.grabExcessHorizontalSpace = false;
		gd7.grabExcessVerticalSpace = false;
		gd7.horizontalAlignment = SWT.LEFT;
		gd7.verticalAlignment = SWT.CENTER;
		label3.setLayoutData(gd7);
		
		notificationServiceCombo = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gd8 = new GridData();
		gd8.grabExcessHorizontalSpace = true;
		gd8.grabExcessVerticalSpace = false;
		gd8.horizontalSpan = 1;
		gd8.horizontalAlignment = SWT.FILL;
		notificationServiceCombo.setLayoutData(gd8);
		
		notificationServiceCombo.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				widgetSelected(e);
			}

			@Override
			public void widgetSelected(SelectionEvent e) {
				setDirty(true);
			}
		});
	}
	
	public String getChannelName() 
	{
		String retVal = null;
		if(this.regExpCheckBox.getSelection()) {
			retVal = this.regExpText.getText();
		} else {
			retVal = this.channelNameCombo.getItem(this.channelNameCombo.getSelectionIndex());
		}
		return retVal;
	}

	public String getNotificationService() {
		return this.notificationServiceCombo.getItem(this.notificationServiceCombo.getSelectionIndex());
	}

	public void setInput(ChannelMappingEditorInput input) 
	{
		this.configuration = input.getChannelMapping().getNotificationServiceMapping().getConfiguration();

		String[] nsItems = NewChannelMappingAction.getNotificationServiceStrings(input.getChannelMapping().
				getNotificationServiceMapping().getConfiguration());
		notificationServiceCombo.setItems(nsItems);
		
		String[] channelNames = getEventChannelStrings();
		channelNameCombo.setItems(channelNames);
		
		String nameString = input.getChannelMapping().getName() != null ? input.getChannelMapping().getName() : "";
		this.regExpText.setText(nameString);
		this.channelNameCombo.select(channelNameCombo.indexOf(nameString));
		this.notificationServiceCombo.select(this.notificationServiceCombo.indexOf(input.getChannelMapping().getNotificationService()));
	}
	
	public void setDirty(boolean dirty) {
		this.dirty = !dirty;
		super.setDirty(dirty);
	}

}
