/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class AlarmDefinitionComposite extends StatusPublishingComposite 
{
	private static final String NOTICE = "Notice";
	private static final String NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP = "No selections made; dialog is only for testing regexp.";
	private Text childFamilyText, childMemberText, childCodeText;
	private Configuration config;
	private ModifyListener childFamilyTextModifyListener, childMemberTextModifyListener, childCodeTextModifyListener; 
	
    /**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AlarmDefinitionComposite(Composite parent, int style, StatusListener statusListener, DirtyListener dirtyListener, Configuration config)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		this.config = config;
		createControl();
	}
	
	public void createControl() 
	{
		FillLayout layout = new FillLayout();
		this.setLayout(layout);
		
		Composite composite = new Composite(this, SWT.NONE);
		composite.setLayout(new GridLayout(3, false));
		
		createFFLabelTextButtonWidgets(composite);
		createFMLabelTextButtonWidgets(composite);
		createFCLabelTextButtonWidgets(composite);
		
		toggleIsComplete();
	}
	
	private void createFFLabelTextButtonWidgets(Composite parent)
	{
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label label = new Label(parent, SWT.NONE);
		label.setText("Child FF: ");
		label.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		childFamilyText = new Text(parent, SWT.BORDER);
		childFamilyText.setLayoutData(gd);
		
		childFamilyTextModifyListener = new ModifyListener() 
		{
			@Override
			public void modifyText(ModifyEvent arg0) {
				toggleIsComplete();
				setDirty(true);
			}
		};
		
		childFamilyText.addModifyListener(childFamilyTextModifyListener);
		
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button childFamilyTestButton = new Button(parent, SWT.PUSH);
		childFamilyTestButton.setText("Test");
		childFamilyTestButton.setLayoutData(gd);
		childFamilyTestButton.addListener(SWT.Selection, new Listener() 
		{
			@Override
			public void handleEvent(Event evt) {
				ElementListSelectionDialog d = new ElementListSelectionDialog(getShell(), new FaultFamilyHitsLabelProvider());
				d.setElements(getFaultFamiliesForRegexp().toArray());
				d.setMultipleSelection(false);
				d.setTitle("Regexp results");
				d.setInitialSelections(new FaultFamily[0]);
				d.open();
				if(d.getReturnCode() == ElementListSelectionDialog.OK) {
					MessageDialog.openInformation(getShell(), NOTICE, NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP);
				}
			}
		});	
	}
	
	private void createFMLabelTextButtonWidgets(Composite parent)
	{
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label label = new Label(parent, SWT.NONE);
		label.setText("Child FM: ");
		label.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		childMemberText = new Text(parent, SWT.BORDER);
		childMemberText.setLayoutData(gd);
		
		childMemberTextModifyListener = new ModifyListener() 
		{
			@Override
			public void modifyText(ModifyEvent arg0) {
				toggleIsComplete();
				setDirty(true);
			}
		};
		
		childMemberText.addModifyListener(childMemberTextModifyListener);
		
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button childMemberTestButton = new Button(parent, SWT.PUSH);
		childMemberTestButton.setText("Test");
		childMemberTestButton.setLayoutData(gd);
		childMemberTestButton.addListener(SWT.Selection, new Listener() 
		{
			@Override
			public void handleEvent(Event evt) {
				ElementListSelectionDialog d = new ElementListSelectionDialog(getShell(), new FaultMemberHitsLabelProvider());
				d.setTitle("Regexp results");
				d.setElements(getFaultMembersForRegexp().toArray());
				d.setMultipleSelection(false);
				d.setInitialSelections(new String[0]);
				d.open();
				if(d.getReturnCode() == ElementListSelectionDialog.OK) {
					MessageDialog.openInformation(getShell(), NOTICE, NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP);
				}
			}
		});	
	}

	private void createFCLabelTextButtonWidgets(Composite parent)
	{
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label label = new Label(parent, SWT.NONE);
		label.setText("Child FC: ");
		label.setLayoutData(gd);

		gd = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		childCodeText = new Text(parent, SWT.BORDER);
		childCodeText.setLayoutData(gd);
		
		childCodeTextModifyListener = new ModifyListener() 
		{
			@Override
			public void modifyText(ModifyEvent arg0) {
				toggleIsComplete();
				setDirty(true);
			}
		};
		
		childCodeText.addModifyListener(childCodeTextModifyListener);
		
		gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Button childCodeTestButton = new Button(parent, SWT.PUSH);
		childCodeTestButton.setText("Test");
		childCodeTestButton.setLayoutData(gd);
		childCodeTestButton.addListener(SWT.Selection, new Listener() 
		{
			@Override
			public void handleEvent(Event evt) {
				ElementListSelectionDialog d = new ElementListSelectionDialog(getShell(), new FaultCodeHitsLabelProvider());
				d.setElements(getFaultCodesForRegexp().toArray());
				d.setMultipleSelection(false);
				d.setTitle("Regexp results");
				d.open();
				if(d.getReturnCode() == ElementListSelectionDialog.OK) {
					MessageDialog.openInformation(getShell(), NOTICE, NO_SELECTIONS_MADE_DIALOG_IS_ONLY_FOR_TESTING_REGEXP);
				}
			}
		});	
	}
	
	private void toggleIsComplete() 
	{
		String parentFcStr = getFaultCode();
		String parentFmStr = getFaultMember();
		if(parentFmStr == null || parentFmStr.trim().length() == 0) {
			this.childCodeText.setEnabled(false);
		} else {
			this.childCodeText.setEnabled(true);
		}
		
		String parentFfStr = getFaultFamily();
		if(parentFfStr == null || parentFfStr.trim().length() == 0) {
			this.childMemberText.setEnabled(false);
		} else {
			this.childMemberText.setEnabled(true);
		}
		
		if( null != parentFfStr && 
			(parentFfStr.length() > 0) &&
			(parentFcStr.length() > 0) &&
			null != parentFmStr &&
			(parentFmStr.length() > 0) )
		{
			setStatus(null);
			notifyListenersOfCompletion(true);
		}
		else {
			setStatus("Please define all fields");
			notifyListenersOfCompletion(false);
		}
	}
	
	public String getFaultFamily() {
		return childFamilyText != null ? childFamilyText.getText().trim() : "";
	}
	
	public String getFaultMember() {
		return childMemberText != null ? childMemberText.getText().trim() : "";
	}
	
	public String getFaultCode() {
		return childCodeText != null ? childCodeText.getText().trim() : "";
	}
	

	private List<FaultFamily> getFaultFamiliesForRegexp() 
	{
		List<FaultFamily> retVal = new ArrayList<FaultFamily>();
		
		try {
			retVal = AlarmConversationUtils.getInstance().findFaultFamiliesByRegexp(this.getFaultFamily(), config);
		} catch(Exception ex) {
			ex.printStackTrace();
		}

		return retVal;
	}

	private Set<String> getFaultMembersForRegexp() 
	{
		Set<String> retVal = new HashSet<String>();
		
		try {
			for(FaultMember member: AlarmConversationUtils.getInstance().findFaultMembersByRegexp(this.getFaultMember(), this.getFaultFamily(), config))
			{
				retVal.add(member.getMemberName());
			}
		} catch(Exception ex) {
			ex.printStackTrace();
		}

		return retVal;
	}
	
	private Set<String> getFaultCodesForRegexp() 
	{
		Set<String> retVal = new HashSet<String>();
		
		try {
			for(FaultCode code: AlarmConversationUtils.getInstance().findFaultCodesByRegexp(this.getFaultCode(), this.getFaultFamily(), config))
			{
				retVal.add(code.getCodeValue().toString());
			}
		} catch(Exception ex) {
			ex.printStackTrace();
		}

		return retVal;
	}
	
	private static class FaultFamilyHitsLabelProvider implements ILabelProvider
	{
		@Override
		public Image getImage(Object arg0) {
			return RcpUtils.getImage("icons/faultfamily.png");
		}

		@Override
		public String getText(Object obj) 
		{
			if(obj instanceof FaultFamily) {
				FaultFamily ff = (FaultFamily) obj;
				return ff.getFamilyName();
			}
			return null;
		}

		@Override
		public void addListener(ILabelProviderListener arg0) {
			// TODO Auto-generated method stub
		}

		@Override
		public void dispose() {
			// TODO Auto-generated method stub
		}

		@Override
		public boolean isLabelProperty(Object arg0,
				String arg1) 
		{
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public void removeListener(
				ILabelProviderListener arg0) 
		{
			// TODO Auto-generated method stub
		}
	}
	
	private static class FaultMemberHitsLabelProvider implements ILabelProvider
	{
		@Override
		public Image getImage(Object arg0) {
			return RcpUtils.getImage("icons/faultmember.png");
		}

		@Override
		public String getText(Object obj) 
		{
			if(obj instanceof String) {
				return (String)obj;
			}
			return null;
		}

		@Override
		public void addListener(ILabelProviderListener arg0) {
			// TODO Auto-generated method stub
		}

		@Override
		public void dispose() {
			// TODO Auto-generated method stub
		}

		@Override
		public boolean isLabelProperty(Object arg0,
				String arg1) 
		{
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public void removeListener(
				ILabelProviderListener arg0) 
		{
			// TODO Auto-generated method stub
		}
	}
	
	private static class FaultCodeHitsLabelProvider implements ILabelProvider
	{
		@Override
		public Image getImage(Object arg0) {
			return RcpUtils.getImage("icons/faultcode.png");
		}

		@Override
		public String getText(Object obj) 
		{
			if(obj instanceof String) {
				return (String)obj;
			}
			return null;
		}

		@Override
		public void addListener(ILabelProviderListener arg0) {
			// TODO Auto-generated method stub
		}

		@Override
		public void dispose() {
			// TODO Auto-generated method stub
		}

		@Override
		public boolean isLabelProperty(Object arg0,
				String arg1) 
		{
			// TODO Auto-generated method stub
			return false;
		}

		@Override
		public void removeListener(
				ILabelProviderListener arg0) 
		{
			// TODO Auto-generated method stub
		}
	}

	public void setReductionLink(ReductionLink reductionLink) 
	{
		if(null != reductionLink) {
			childFamilyText.removeModifyListener(childFamilyTextModifyListener);
			childMemberText.removeModifyListener(childMemberTextModifyListener);
			childCodeText.removeModifyListener(childCodeTextModifyListener);
			
			this.childFamilyText.setText(reductionLink.getAlarmDefinitionByChildalarmdefid().getFaultFamily());
			this.childMemberText.setText(reductionLink.getAlarmDefinitionByChildalarmdefid().getFaultMember());
			this.childCodeText.setText(reductionLink.getAlarmDefinitionByChildalarmdefid().getFaultCode());
			
			childFamilyText.addModifyListener(childFamilyTextModifyListener);
			childMemberText.addModifyListener(childMemberTextModifyListener);
			childCodeText.addModifyListener(childCodeTextModifyListener);
			
			toggleIsComplete();
		}
	}
}
