/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.widgets;

import java.util.Set;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdbgui.widgets.StatusPublishingComposite;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;
import alma.obops.tmcdbgui.widgets.support.StatusListener;

public class AlarmCategoryAttributesComposite extends StatusPublishingComposite 
{
	public static final String ALARMCATEGORY_ALREADY_EXISTS = "Alarm category already exists: name must be unique";

	private Text categoryNameText;
	private Text categoryDescriptionText;
	private Text categoryPathText;
	private Button defaultButton;
	private Set<AlarmCategory> existingAlarmCategories;
	private Configuration configuration;
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 * @param ac the alarm category that is being "dealt" with.
	 */
	public AlarmCategoryAttributesComposite(Composite parent, int style, AlarmCategory ac, StatusListener statusListener, DirtyListener dirtyListener, Configuration config)
	{
		super(parent, style);
		this.addStatusListener(statusListener);
		this.addDirtyListener(dirtyListener);
		this.configuration = ConfigurationHelper.findConfiguration(config);
		createControl(ac);
	}
	
	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AlarmCategoryAttributesComposite(Composite parent, int style, DirtyListener dirtyListener, Configuration config) 
	{
		this(parent, style, null, null, dirtyListener, config);
	}

	/**
	 * Constructor.
	 * @param parent the parent composite.
	 * @param style the style for the widget.
	 */
	public AlarmCategoryAttributesComposite(Composite parent, int style, StatusListener statusListener, Configuration config) 
	{
		this(parent, style, null, statusListener, null, config);
	}

	private void createControl(AlarmCategory ac) 
	{
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;  // label, entry
		setLayout( layout );
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true, 1, 1);
		setLayoutData(gridData);

		createCategoryNameControl();
		createCategoryDescriptionControl();
		createCategoryPathControl();
		creatDefaultControl();
        setAlarmCategory(ac);
        addKeyListeners();
	}	
	
	private void createCategoryPathControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Path");
		categoryPathText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		categoryPathText.setLayoutData(gridData);
		categoryPathText.addKeyListener(new SetDirtyKeyListener());
	}

	private void creatDefaultControl() 
	{
		Label label = new Label(this, SWT.NONE);
		label.setText("Is default?");
		this.defaultButton = new Button(this, SWT.CHECK);
		defaultButton.addKeyListener(new SetDirtyKeyListener());
		defaultButton.addMouseListener(new MouseListener() 
		{
			@Override
			public void mouseDoubleClick(MouseEvent evt) {
				setDirty(true);
			}

			@Override
			public void mouseDown(MouseEvent evt) {
				setDirty(true);
			}

			@Override
			public void mouseUp(MouseEvent evt) {
				setDirty(true);
			}
		});
	}

	private void createCategoryDescriptionControl() {
		Label label = new Label(this, SWT.NONE);
		label.setText("Description");
		categoryDescriptionText = new Text(this, SWT.BORDER);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		categoryDescriptionText.setLayoutData(gridData);
		categoryDescriptionText.addKeyListener(new SetDirtyKeyListener());
	}

	private void createCategoryNameControl() 
	{
        Label label = new Label(this, SWT.NONE);
        label.setText("Name");
        categoryNameText = new Text(this, SWT.BORDER);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        categoryNameText.setLayoutData(gridData);
        categoryNameText.addKeyListener(new SetDirtyKeyListener());
	}

	private void addKeyListeners() 
	{
		// At each keystroke computes whether this page is complete
		KeyListener completionKL = new KeyListener() 
		{
			public void keyPressed( KeyEvent e ) {
				// ignore
			}

			public void keyReleased( KeyEvent e ) {
				isComplete();
			}
		};
		categoryNameText.addKeyListener(completionKL);
		categoryDescriptionText.addKeyListener(completionKL);
		categoryPathText.addKeyListener(completionKL);
	}
	
	private boolean alarmCategoryExistsInConfig()
	{
		boolean retVal = false;
		this.existingAlarmCategories = configuration.getAlarmCategories();
		
		try	{
			retVal = foundCorrespondingAlarmCategory();
		}
		catch(Exception ex) {
			throw new RuntimeException("Unable to get the alarm categories for the configuration", ex);
		}
		
		if(retVal == true) {
			this.setStatus(ALARMCATEGORY_ALREADY_EXISTS);
		} else {
			this.setStatus(null);
		}
		return retVal;
	}
	
	private boolean foundCorrespondingAlarmCategory() {
		boolean retVal = false;
		for(AlarmCategory ac: existingAlarmCategories) 
		{
			if(ac.getAlarmCategoryName().equals(this.getCategoryName()))
			{
				retVal = true;
				break;
			}
		}
		return retVal;
	}
	
	public String getCategoryName() {
		return this.categoryNameText.getText().trim();
	}

	/** @return <code>true</code> when all required fields are populated */
	public boolean isComplete() 
	{	
		boolean complete = 
			!alarmCategoryExistsInConfig() &&
			(categoryNameText.getText().length() > 0) &&
			(categoryPathText.getText().length() > 0) &&
			(categoryDescriptionText.getText().length() > 0);

		notifyListenersOfCompletion(complete);
		return complete;
	}

	private class SetDirtyKeyListener implements KeyListener 
	{
		@Override
		public void keyPressed(KeyEvent e) {
			setDirty(true);
		}

		@Override
		public void keyReleased(KeyEvent e) {
		}
	}

	public void setAlarmCategory(AlarmCategory ac) 
	{
		if(null == ac) 
		{
			return;
		}
		
		this.categoryNameText.setText(ac.getAlarmCategoryName());
		String catDescription = ac.getDescription() == null ? "" : ac.getDescription();
		this.categoryDescriptionText.setText(catDescription);
		String catPath = ac.getPath() == null ? "" : ac.getPath();
		this.categoryPathText.setText(catPath);
		this.defaultButton.setSelection(ac.getIsDefault());
	}

	public String getCategoryDescription() {
		return this.categoryDescriptionText.getText();
	}
	
	public void setConfiguration(Configuration config)
	{
		this.configuration = config;
	}

	public String getCategoryPath() {
		return this.categoryPathText.getText().trim();
	}

	public boolean isDefault() {
		return defaultButton.getSelection();
	}
}
