/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.views.sorters;

import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionLinkType;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmCategoryList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmDefinitionList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.DefaultMemberList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultCodeList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultFamilyList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultMemberList;

public class AlarmCategoriesTreeSorter extends ViewerSorter
{
	@Override
	public int compare(Viewer viewer, Object obj1, Object obj2)
	{
		int retVal = 0;
		
		// TODO: should we have the domain objects implement comparable interface?
		//       it would make things simpler...
		if(obj1 instanceof AlarmCategory)
		{
			retVal = compareAlarmCategoryTo((AlarmCategory)obj1, obj2);
		}
		else if(obj1 instanceof FaultFamily)
		{
			retVal = compareFaultFamilyTo((FaultFamily)obj1, obj2);	
		}
		else if(obj1 instanceof FaultMember)
		{
			retVal = compareFaultMemberTo((FaultMember)obj1, obj2);	
		}
		else if(obj1 instanceof FaultCode)
		{
			retVal = compareFaultCodeTo((FaultCode)obj1, obj2);
		}
		else if(obj1 instanceof AlarmCategoryList)
		{
			retVal = compareAlarmCategoryListTo((AlarmCategoryList)obj1, obj2);
		}
		else if(obj1 instanceof AlarmDefinition)
		{
			retVal = compareAlarmDefinitionTo((AlarmDefinition)obj1, obj2);	
		}
		else if(obj1 instanceof AlarmDefinitionList)
		{
			retVal = compareAlarmDefinitionListTo((AlarmDefinitionList)obj1, obj2);	
		}
		else if(obj1 instanceof DefaultMemberList)
		{
			retVal = compareDefaultMemberListTo((DefaultMemberList)obj1, obj2);	
		}
		else if(obj1 instanceof FaultCodeList)
		{
			retVal = compareFaultCodeListTo((FaultCodeList)obj1, obj2);	
		}
		else if(obj1 instanceof FaultFamilyList)
		{
			retVal = compareFaultFamilyListTo((FaultFamilyList)obj1, obj2);	
		}
		else if(obj1 instanceof FaultMemberList)
		{
			retVal = compareFaultMemberListTo((FaultMemberList)obj1, obj2);	
		}
		else if(obj1 instanceof ReductionLink)
		{
			retVal = compareReductionLinkTo((ReductionLink)obj1, obj2);
		}
		
		return retVal;
	}

	private int compareAlarmDefinitionTo(AlarmDefinition ad1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof AlarmDefinition)
		{
			AlarmDefinition ad2 = (AlarmDefinition)obj2; 
			try {
				ad1 = AlarmDefinitionHelper.findAlarmDefinition(ad1);
				ad2 = AlarmDefinitionHelper.findAlarmDefinition(ad2);
			} catch(Exception ex) {
				ex.printStackTrace();
				throw new RuntimeException("Cannot hydrate alarm definitions", ex);
			}
			// special hack to handle sorting in semi-alphabetic manner (we want the strings alphabetical but for the
			// end part (faultcode) which should be sorted numerically - so e.g. <FF, FM, FC> where FC is ordered by
			// increasing numbers for FF/FM pairs.
			
			String str1 = ad1.getFaultFamily() + ad1.getFaultMember() + ad1.getFaultCode(); 
			String str2 = ad2.getFaultFamily() + ad2.getFaultMember() + ad2.getFaultCode();
			retVal = str1.compareTo(str2);	
		}
		return retVal;
	}

	private int compareFaultFamilyListTo(FaultFamilyList obj1, Object obj2) 
	{
		int retVal = 0;
		
		if(obj2 instanceof FaultFamilyList) 
		{
			FaultFamilyList fmList = (FaultFamilyList) obj2;
			retVal = obj1.get(0).getConfiguration().getConfigurationId().compareTo(fmList.get(0).getConfiguration().getConfigurationId());
		}

		return retVal;
	}

	private int compareAlarmCategoryTo(AlarmCategory obj1, Object obj2) {
		int retVal = 0;
		if(obj2 instanceof AlarmCategory)
		{
			AlarmCategory ac2 = (AlarmCategory)obj2;
			retVal = obj1.getAlarmCategoryName().compareTo(ac2.getAlarmCategoryName());	
		}
		return retVal;
	}

	private int compareFaultFamilyTo(FaultFamily obj1, Object obj2) {
		int retVal = 0;
		if(obj2 instanceof FaultFamily)
		{
			FaultFamily ff2 = (FaultFamily)obj2;
			retVal = obj1.getFamilyName().compareTo(ff2.getFamilyName());	
		}
		return retVal;
	}

	private int compareFaultMemberTo(FaultMember obj1, Object obj2) {
		int retVal = 0;
		if(obj2 instanceof FaultMember)
		{
			FaultMember fm2 = (FaultMember)obj2;
			retVal = obj1.getMemberName().compareTo(fm2.getMemberName());	
		}
		return retVal;
	}

	private int compareFaultMemberListTo(FaultMemberList obj1, Object obj2) {
		int retVal = 0;
		
		if(obj2 instanceof FaultMemberList) 
		{
			FaultMemberList fmList = (FaultMemberList) obj2;
			retVal = obj1.get(0).getFaultFamily().getFamilyName().compareTo(fmList.get(0).getFaultFamily().getFamilyName());
		}

		return retVal;
	}

	private int compareFaultCodeListTo(FaultCodeList obj1, Object obj2) {
		int retVal = 0;
		
		if(obj2 instanceof FaultCodeList) 
		{
			FaultCodeList fcList = (FaultCodeList) obj2;
			retVal = obj1.get(0).getFaultFamily().getFamilyName().compareTo(fcList.get(0).getFaultFamily().getFamilyName());
		}

		return retVal;
	}

	private int compareDefaultMemberListTo(DefaultMemberList obj1, Object obj2) {
		int retVal = 0;
		
		if(obj2 instanceof DefaultMemberList) 
		{
			DefaultMemberList dfmList = (DefaultMemberList) obj2;
			retVal = obj1.get(0).getFaultFamily().getFamilyName().compareTo(dfmList.get(0).getFaultFamily().getFamilyName());
		}

		return retVal;
	}

	private int compareAlarmDefinitionListTo(AlarmDefinitionList adList1,
			Object obj2) 
	{
		int retVal = 0;
		
		if(obj2 instanceof AlarmDefinitionList) 
		{
			AlarmDefinitionList adList2 = (AlarmDefinitionList) obj2;
			Integer adListOneStart = adList1.getStart();
			Integer adListTwoStart = adList2.getStart();
			retVal = adListOneStart.compareTo(adListTwoStart);
		}

		return retVal;
	}

	private int compareFaultCodeTo(FaultCode obj1, Object obj2) {
		int retVal = 0;
		if(obj2 instanceof FaultCode)
		{
			FaultCode fc2 = (FaultCode)obj2;
			retVal = obj1.getCodeValue().compareTo(fc2.getCodeValue());	
		}
		return retVal;
	}
	
	private int compareReductionLinkTo(ReductionLink redLink1, Object obj2) 
	{
		int retVal = 0;
		if(obj2 instanceof ReductionLink)
		{
			ReductionLink redLink2 = (ReductionLink) obj2;
			redLink1.setAlarmDefinitionByChildalarmdefid(AlarmDefinitionHelper.findAlarmDefinition(redLink1.getAlarmDefinitionByChildalarmdefid()));
			redLink2.setAlarmDefinitionByChildalarmdefid(AlarmDefinitionHelper.findAlarmDefinition(redLink2.getAlarmDefinitionByChildalarmdefid()));
			redLink1.setAlarmDefinitionByParentalarmdefid(AlarmDefinitionHelper.findAlarmDefinition(redLink1.getAlarmDefinitionByParentalarmdefid()));
			redLink2.setAlarmDefinitionByParentalarmdefid(AlarmDefinitionHelper.findAlarmDefinition(redLink2.getAlarmDefinitionByParentalarmdefid()));
			if(redLink1.getType().equals(ReductionLinkType.NODE) && redLink2.getType().equals(ReductionLinkType.MULTIPLICITY))
			{
				retVal = -1;
			} 
			else if(redLink1.getType().equals(ReductionLinkType.MULTIPLICITY) && redLink2.getType().equals(ReductionLinkType.NODE)) 
			{
				retVal = 1;
			}
			else if(redLink1.getType().equals(ReductionLinkType.NODE)) 
			{
				String text1 = AlarmDefinitionHelper.getNameText(redLink1.getAlarmDefinitionByChildalarmdefid());
				String text2 = AlarmDefinitionHelper.getNameText(redLink2.getAlarmDefinitionByChildalarmdefid());
				retVal = text1.compareTo(text2);
			}
			else if(redLink1.getType().equals(ReductionLinkType.MULTIPLICITY)) 
			{
				String text1 = AlarmDefinitionHelper.getNameText(redLink1.getAlarmDefinitionByParentalarmdefid());
				String text2 = AlarmDefinitionHelper.getNameText(redLink2.getAlarmDefinitionByParentalarmdefid());
				retVal = text1.compareTo(text2);
			}
		}
		
		return retVal;
	}

	private int compareAlarmCategoryListTo(AlarmCategoryList obj1, Object obj2) {
		int retVal = 0;
		
		if(obj2 instanceof AlarmCategoryList) 
		{
			AlarmCategoryList acList = (AlarmCategoryList) obj2;
			retVal = obj1.get(0).getAlarmCategoryName().compareTo(acList.get(0).getAlarmCategoryName());
		}

		return retVal;
	}
}
