/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.views;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.DecoratingStyledCellLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.part.ViewPart;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.actions.add.AddAlarmCategoryAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddDefaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultCodeAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultFamilyAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddFaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddMultiplicityReductionLinkAction;
import alma.obops.tmcdb.alarms.ui.actions.add.AddNodeReductionLinkAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteAlarmCategoryAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteDefaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteFaultCodeAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteFaultFamilyAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteFaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.delete.DeleteReductionLinkAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditAlarmCategoryAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditDefaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditFaultCodeAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditFaultFamilyAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditFaultMemberAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditReductionLinkAction;
import alma.obops.tmcdb.alarms.ui.actions.edit.EditReductionThresholdAction;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmCategoryHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ContactHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.DefaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultCodeHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionLinkHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionThresholdHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmCategoryList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.DefaultMemberList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultCodeList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultMemberList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.comparers.AlarmCategoriesTreeComparer;
import alma.obops.tmcdb.alarms.ui.views.providers.AlarmCategoriesTreeContentsProvider;
import alma.obops.tmcdb.alarms.ui.views.providers.AlarmCategoriesTreeLabelProvider;
import alma.obops.tmcdb.alarms.ui.views.sorters.AlarmCategoriesTreeSorter;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.views.ConfigurationsView;
import alma.tmcdb.domain.HwConfiguration;

public class AlarmCategoriesView extends ViewPart implements IModelChangeListener
{
	public static final String ID = "alarm-categories.view";
	private TreeViewer treeViewer;
	private AddFaultFamilyAction addFaultFamilyAction;
	private AddAlarmCategoryAction addAlarmCategoryAction;
	private AddNodeReductionLinkAction addNodeReductionLinkAction;
	private AddMultiplicityReductionLinkAction addMultiplicityReductionLinkAction;
	private AddFaultMemberAction addFaultMemberAction;
	private AddFaultCodeAction addFaultCodeAction;
	private AddDefaultMemberAction addDefaultMemberAction;
	
	private EditAlarmCategoryAction editAlarmCategoryAction;
	private EditDefaultMemberAction editDefaultMemberAction;
	private EditFaultFamilyAction editFaultFamilyAction;
	private EditFaultMemberAction editFaultMemberAction;
	private EditFaultCodeAction editFaultCodeAction;
	private EditReductionLinkAction editNodeReductionLinkAction;
	private EditReductionThresholdAction editReductionThresholdAction;
	
	private DeleteAlarmCategoryAction deleteAlarmCategoryAction;
	private DeleteFaultFamilyAction deleteFaultFamilyAction;
	private DeleteFaultMemberAction deleteFaultMemberAction;
	private DeleteDefaultMemberAction deleteDefaultMemberAction;
	private DeleteFaultCodeAction deleteFaultCodeAction;
	private DeleteReductionLinkAction deleteReductionLinkAction;
	
	private AlarmCategoriesTreeContentsProvider contentsProvider;
	private AlarmCategoriesTreeSorter sorter;
	private AlarmCategoriesTreeComparer comparer;
	private HwConfiguration configuration;
	
	@Override
	public void createPartControl(Composite parent) 
	{
		int style = SWT.BORDER | SWT.FULL_SELECTION ;
		treeViewer = new TreeViewer(parent, style);
		this.contentsProvider = new AlarmCategoriesTreeContentsProvider();
		treeViewer.setContentProvider(contentsProvider);
		this.sorter =  new AlarmCategoriesTreeSorter();
		treeViewer.setSorter(sorter);
		this.comparer = new AlarmCategoriesTreeComparer();
		treeViewer.setComparer(comparer);
		
		treeViewer.getTree().setLinesVisible( true );
		treeViewer.getTree().setHeaderVisible( true );
        
        // First column -- name and icon for all tree nodes
        TreeViewerColumn col0 = new TreeViewerColumn( treeViewer, SWT.NONE );
        col0.getColumn().setWidth( 150 );
        col0.getColumn().setMoveable( false );
        col0.getColumn().setText( "Name" );
        DecoratingStyledCellLabelProvider lp = new DecoratingStyledCellLabelProvider(new AlarmCategoriesTreeLabelProvider(0), getSite().getWorkbenchWindow().getWorkbench().getDecoratorManager(), null);
        col0.setLabelProvider(lp);

    	// Register our viewer as a source of selections
        getSite().setSelectionProvider( treeViewer );

		makeActions();
		makeContextMenu();
	}

	public void setInput( HwConfiguration conf ) 
    {
		this.configuration = conf;
		try {
			new ProgressMonitorDialog(this.getSite().getShell()).
				   run(true, false, new ThreadForSettingInput());
		} catch (InvocationTargetException e) {
			e.printStackTrace();
			throw new RuntimeException("Could not invoke hydration thread", e);
		} catch (InterruptedException e) {
			e.printStackTrace();
			throw new RuntimeException("Hydration thread was interrupted", e);
		}
		
		try {
			Object[] elements = treeViewer.getExpandedElements();
			TreePath[] treePaths = treeViewer.getExpandedTreePaths();    		
			treeViewer.getTree().setRedraw(false);
			Object[] input = new Object[1];
			input[0] = configuration;
			treeViewer.setInput( input );
			treeViewer.setExpandedElements(elements);
			treeViewer.setExpandedTreePaths(treePaths);
		}
		finally {
			treeViewer.getTree().setRedraw(true);
		}
    }
	
	private class ThreadForSettingInput implements IRunnableWithProgress 
	{
		/**
		 * LongRunningOperation constructor
		 * 
		 * @param indeterminate whether the animation is unknown
		 */
		public ThreadForSettingInput() {
		}

		/**
		 * Runs the long running operation
		 * 
		 * @param monitor the progress monitor
		 */
		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException 
		{
			monitor.beginTask("Loading...", IProgressMonitor.UNKNOWN);
			try 
			{
				ConfigurationHelper.findConfiguration(configuration.getSwConfiguration());
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not hydrate alarms", e);
			}
			monitor.done();
		}
	}
	
	private void makeContextMenu() 
	{
		final MenuManager mgr = new MenuManager("alarmsPopup");
        mgr.setRemoveAllWhenShown(true);
        mgr.addMenuListener( new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				fillContextMenu(manager);
			}
		});
        Control ctrl = treeViewer.getControl();
        ctrl.setMenu( mgr.createContextMenu( ctrl ));
        getSite().registerContextMenu(mgr, treeViewer);
	}

	private void makeActions() 
	{
		// the following actions are available only to 'god' user (almamgr)
		if(GuiUtils.isGodUser())
		{
			IWorkbenchWindow win = getSite().getWorkbenchWindow();
			
			addAlarmCategoryAction = new AddAlarmCategoryAction( win, null );
			addFaultFamilyAction = new AddFaultFamilyAction( win, null, null );
			addFaultMemberAction = new AddFaultMemberAction( win, null, null );
			addDefaultMemberAction = new AddDefaultMemberAction( win );
			addFaultCodeAction = new AddFaultCodeAction( win, null, null );
			addNodeReductionLinkAction = new AddNodeReductionLinkAction( win, null, null );
			addMultiplicityReductionLinkAction = new AddMultiplicityReductionLinkAction( win, null, null );
			
			editAlarmCategoryAction = new EditAlarmCategoryAction(win, this);
			editFaultFamilyAction = new EditFaultFamilyAction(win, this);
			editFaultMemberAction = new EditFaultMemberAction(win, this);
			editDefaultMemberAction = new EditDefaultMemberAction(win, this);
			editFaultCodeAction = new EditFaultCodeAction(win, this);
			editNodeReductionLinkAction = new EditReductionLinkAction(win, this);
			editReductionThresholdAction = new EditReductionThresholdAction(win, this);
			
			deleteAlarmCategoryAction = new DeleteAlarmCategoryAction(win);
			deleteFaultFamilyAction = new DeleteFaultFamilyAction(win);
			deleteFaultMemberAction = new DeleteFaultMemberAction(win);
			deleteDefaultMemberAction = new DeleteDefaultMemberAction(win);
			deleteFaultCodeAction = new DeleteFaultCodeAction(win);
			deleteReductionLinkAction = new DeleteReductionLinkAction(win);
		}
		
		// Double-click support for opening the editors
    	treeViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {

				if( event.getSelection() instanceof IStructuredSelection ) 
				{
					IStructuredSelection selection = (IStructuredSelection)event.getSelection();
					if( selection.getFirstElement() instanceof FaultFamily && GuiUtils.isGodUser()) {
						editFaultFamilyAction.selectionChanged(AlarmCategoriesView.this, selection);
						editFaultFamilyAction.run();
					} else if(selection.getFirstElement() instanceof FaultMember && GuiUtils.isGodUser()) {
						editFaultMemberAction.selectionChanged(AlarmCategoriesView.this, selection);
						editFaultMemberAction.run();
					} else if(selection.getFirstElement() instanceof DefaultMember && GuiUtils.isGodUser()) {
						editDefaultMemberAction.selectionChanged(AlarmCategoriesView.this, selection);
						editDefaultMemberAction.run();
					} else if(selection.getFirstElement() instanceof FaultCode && GuiUtils.isGodUser()) {
						editFaultCodeAction.selectionChanged(AlarmCategoriesView.this, selection);
						editFaultCodeAction.run();
					} else if(selection.getFirstElement() instanceof AlarmCategory && GuiUtils.isGodUser()) {
						editAlarmCategoryAction.selectionChanged(AlarmCategoriesView.this, selection);
						editAlarmCategoryAction.run();
					} 
					else if(selection.getFirstElement() instanceof ReductionLink && GuiUtils.isGodUser()) {
						editNodeReductionLinkAction.selectionChanged(AlarmCategoriesView.this, selection);
						editNodeReductionLinkAction.run();	
					} else if(selection.getFirstElement() instanceof ReductionThreshold && GuiUtils.isGodUser()) {
						editReductionThresholdAction.selectionChanged(AlarmCategoriesView.this, selection);
						editReductionThresholdAction.run();
					}
				}
			}
    	});
	}

	@Override
	public void setFocus() {
	}
	
	private void fillContextMenu(IMenuManager manager) 
	{
		if(!GuiUtils.isGodUser())
		{
			return;
		}

		ISelection selection = treeViewer.getSelection();

		if( selection instanceof IStructuredSelection ) 
		{
			manager.add(addNodeReductionLinkAction);
			manager.add(addMultiplicityReductionLinkAction);
			IStructuredSelection sselection = (IStructuredSelection) selection;
			Object selNode = sselection.getFirstElement();

			if( selNode instanceof Configuration) {
				manager.add(addAlarmCategoryAction);
			} else if(selNode instanceof AlarmCategoryList) {
				manager.add(addAlarmCategoryAction);
			}
			else if( selNode instanceof HwConfiguration) {
				manager.add(addAlarmCategoryAction);
			}
			else if( selNode instanceof AlarmCategory) {
				manager.add(addFaultFamilyAction);
				manager.add(editAlarmCategoryAction);
				manager.add(deleteAlarmCategoryAction);
			} 
			else if( selNode instanceof FaultFamily) {
				manager.add(addFaultMemberAction);
				manager.add(addDefaultMemberAction);
				if( ((FaultFamily)selNode).getDefaultMembers().isEmpty() ) 
				{
					addDefaultMemberAction.setEnabled(true);
				} else {
					addDefaultMemberAction.setEnabled(false);
				}
				manager.add(addFaultCodeAction);
				manager.add(editFaultFamilyAction);
				manager.add(deleteFaultFamilyAction);
			} else if( selNode instanceof FaultMember ) {
				manager.add(editFaultMemberAction);
				manager.add(deleteFaultMemberAction);
			} else if( selNode instanceof DefaultMember ) {
				manager.add(editDefaultMemberAction);
				manager.add(deleteDefaultMemberAction);
			} 
			else if( selNode instanceof FaultCode ) {
				manager.add(editFaultCodeAction);
				manager.add(deleteFaultCodeAction);
			}
			else if( selNode instanceof FaultMemberList ) {
				manager.add(addFaultMemberAction);
			} else if( selNode instanceof DefaultMemberList) {
				manager.add(addDefaultMemberAction);
				if(((DefaultMemberList)selNode).size() > 0 ) {
					addDefaultMemberAction.setEnabled(false);
				} else {
					addDefaultMemberAction.setEnabled(true);
				}
			} else if( selNode instanceof FaultCodeList ) {
				manager.add(addFaultCodeAction);
			} else if( selNode instanceof ReductionLink ) {
				manager.add(editNodeReductionLinkAction);
				manager.add(deleteReductionLinkAction);
			} else if( selNode instanceof ReductionThreshold ) {
				manager.add(editReductionThresholdAction);
			}
		}
	}
	
	public void refreshTreeAndMaintainSelection()
	{
		Object[] elements = treeViewer.getExpandedElements();
		TreePath[] treePaths = treeViewer.getExpandedTreePaths();
		treeViewer.refresh();
		treeViewer.setExpandedTreePaths(treePaths);
		treeViewer.setExpandedElements(elements);
	}
	
	public void setConfiguration(HwConfiguration conf) {
		this.configuration = conf;
		Configuration swconfig = ConfigurationHelper.findConfiguration(conf.getSwConfiguration());
		this.contentsProvider.setConfiguration(swconfig);
		this.addAlarmCategoryAction.setConfiguration(swconfig);
		this.editAlarmCategoryAction.setConfiguration(swconfig);
		this.addFaultFamilyAction.setConfiguration(swconfig);
		this.addMultiplicityReductionLinkAction.setConfiguration(swconfig);
		this.addNodeReductionLinkAction.setConfiguration(swconfig);
	}

	@Override
	public void externalModelChange() 
	{
		clearCaches();
		ConfigurationsView hwView = (ConfigurationsView) RcpUtils.findView(ConfigurationsView.ID);
		hwView.externalModelChange();
		HwConfiguration[] hwConfigs = hwView.getInput();
		for(HwConfiguration hwconf : hwConfigs) 
		{
			if(hwconf.getId().equals(this.configuration.getId())) 
			{
				this.setConfiguration(hwconf);
				this.setInput(hwconf);
				break;
			}
		}
	}

	@Override
	public void internalModelChange() {
		clearCaches();
		this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
		this.refreshTreeAndMaintainSelection();
		this.getSite().getShell().setCursor(null);
	}

	public void clearCaches() 
	{
		AlarmCategoryHelper.clearCache();
		AlarmDefinitionHelper.clearCache();
		ContactHelper.clearCache();
		DefaultMemberHelper.clearCache();
		FaultFamilyHelper.clearCache();
		FaultMemberHelper.clearCache();
		FaultCodeHelper.clearCache();
		LocationHelper.clearCache();
		ReductionLinkHelper.clearCache();
		ReductionThresholdHelper.clearCache();
//		ConfigurationHelper.clearCache();
	}
}
