/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * RcplUtils.java
 * 
 * Copyright European Southern Observatory 2008
 */

package alma.obops.tmcdb.alarms.ui.utils;

import java.io.File;
import java.net.URL;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.application.IWorkbenchWindowConfigurer;

/**
 * A collection of utility methods
 *
 * @author amchavan, Sep 5, 2008
 * 
 */



public class RcpUtils {

    private static ImageRegistry image_registry;
    private static Clipboard clipboard;
    
    /** The configurer for the main application window */
    private static IWorkbenchWindowConfigurer windowConfigurer;

    /** @param configurer The configurer for the main application window */
    public static void setWindowConfigurer( IWorkbenchWindowConfigurer configurer ) {
        windowConfigurer = configurer;
    }
//    
//    public static URL newURL( String url_name ) {
//        try {
//            return new URL( url_name );
//        }
//        catch( MalformedURLException e ) {
//            throw new RuntimeException( "Malformed URL " + url_name, e );
//        }
//    }

    /**
     * @return Our image registry; if needed it creates one first
     */
    private static ImageRegistry getImageRegistry() {
        if( image_registry == null ) {
            image_registry = new ImageRegistry();
        }
        return image_registry;
    }

    /**
     * Looks up an image in the registry: if not found, will attempt to
     * load the corresponding resource from the classpath and store the
     * corresponding image in the registry.
     * 
     * @param filename  Name of the file defining the image
     * 
     * @return An image (what else?)
     */
    // TODO -- harmonize getImage() and getImageDescriptor()
    public static Image getImage( String filename ) {
        Image image = getImageRegistry().get( filename );
        if( image == null ) {
            ClassLoader cl = RcpUtils.class.getClassLoader();
            
            URL url = cl.getResource( filename );
            image = ImageDescriptor.createFromURL( url ).createImage();
            image_registry.put( filename, image );
        }
        return image;
    }

    /**
     * Looks up an image descriptor in the registry: if not found, will attempt
     * to load the corresponding resource from the classpath and store the
     * corresponding descriptor in the registry.
     * 
     * @param filename Name of the file defining the image
     * 
     * @return An image (what else?)
     */
    // TODO -- harmonize getImage() and getImageDescriptor()
    public static ImageDescriptor getImageDescriptor( String filename ) {
        ImageDescriptor image = getImageRegistry().getDescriptor( filename );
        if( image == null ) {
            ClassLoader cl = RcpUtils.class.getClassLoader();
            
            URL url = cl.getResource( filename );
            image = ImageDescriptor.createFromURL( url );
            image_registry.put( filename, image );
        }
        return image;
    }

    /** Scale the input image to the given width and height */
    public static Image scaleImage( Image image, int width, int height ) {
        
        ImageData original = image.getImageData();
        ImageData scaled = original.scaledTo( width, height );
        Image ret = new Image( Display.getCurrent(), scaled );
        return ret;
    }
    
    /** @return One of the RCP's shared images */
    public static Image getSharedImage( String key ) {
        return PlatformUI.getWorkbench().getSharedImages().getImage( key );
    }
    
    public static Clipboard getClipboard() {
        if( clipboard == null ) {
            clipboard = new Clipboard( Display.getCurrent() );
        }
        return clipboard;
    }

    public static String getExtension( File file ) {
        String[] temp = file.getName().replace( File.separator, "/" ).split( "/" );
        String basename = temp[ temp.length - 1 ];
        temp = basename.split( "\\." );
        String extension = temp[ temp.length - 1 ];
        return extension;
    }

    /**
     * @return <code>true</code> is the input file contains an image,
     *         <code>false</code> otherwise. The test is based on the filename's
     *         extension.
     */
    public static boolean isImageFile( File file ) {
        String extension = getExtension( file ).toLowerCase();
        if(        extension.equals( "gif" ) 
                || extension.equals( "png" ) 
                || extension.equals( "jpg" )
                || extension.equals( "jpeg" )
                || extension.equals( "tiff" )
                || extension.equals( "bmp" ) ) {
            return true;
        }
        return false;
    }
    
    public static void statusMessage( String message ) {

    	final String fMessage = message;
        final IStatusLineManager status = windowConfigurer.getActionBarConfigurer().getStatusLineManager();
        windowConfigurer.getWindow().getShell().getDisplay().asyncExec(new Runnable() {
			public void run() {
				status.setMessage( fMessage );
			}
		});
        
    }
  
    /**
     * Lookup a view by its ID.
     * @param viewID    ID of the desired IViewPart
     */
    public static IViewPart findView( String viewID ) {
        return findView(viewID, null, false);
    }

    /**
     * Lookup a view by its ID.
     * @param viewID    ID of the desired IViewPart
     * @param focus     Whether we want to have the found view focused or not
     */
    public static IViewPart findView( String viewID, boolean focus ) {
        return findView(viewID, null, focus);
    }

    /**
     * Lookup a view by its ID.
     * @param viewID    ID of the desired IViewPart
     * @param focus     Whether we want to have the found view focused or not
     */
    public static IViewPart findView( String viewID, IWorkbenchPage page ) {
        return findView(viewID, page, false);
    }

    /**
     * Lookup a view by its ID, and within a given page context. This is recommended by several people,
     * look in http://dev.eclipse.org/newslists/news.eclipse.platform.rcp/msg33375.html
     * 
     * @param viewID    ID of the desired IViewPart
     * @param page      The Workbench page where to look for the view.
     * @param focus     Whether we want to have the found view focused or not
     * @throws PartInitException If the desired IViewPart has some error while initializing
     */
    public static IViewPart findView( String viewID , IWorkbenchPage page , boolean focus) {

    	IWorkbenchPage p = page;
    	if( p == null )
    	{
    		if(null != PlatformUI.getWorkbench() && null != PlatformUI.getWorkbench().getActiveWorkbenchWindow())
    		{
    			p = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
    		}
    	}
        if( p != null ) 
        {
        	if( p.findViewReference( viewID ) == null ) {
				try {
					p.showView( viewID );
				} catch (PartInitException e) {
					errorMessage(e, page.getWorkbenchWindow().getShell(), "Cannot open view",
					"An unexpected error ocurred while trying to open the '" + viewID + "' view");
				}
        	}
        	IViewPart part =  p.findViewReference( viewID ).getView( true );
        	if( focus )
        		p.activate(part);
        	return part;
        }
        return null;
    }

    /**
     * Displays an info popup.
     * 
     * @param shell     The top-level shell
     * @param title     Title of the popup window
     * @param message   Text of the information message
     */
    public static void infoMessage( 
            final Shell shell, 
            final String title,
            final String message ) {
    	shell.getDisplay().asyncExec(new Runnable() {
        	public void run() {
        		MessageDialog.openInformation(shell, title, message);
        	}
        });
    }

    /**
     * Display an error popup.
     * 
     * @param e         The exception describing the problem, cannot be null
     * @param shell     The top-level shell
     * @param title     Title of the popup window
     * @param message   Text of the error message
     */
    public static void errorMessage( final Exception e, 
                              final Shell shell, 
                              final String title,
                              final String message ) {
        final IStatus status = new Status( IStatus.ERROR, "TmcdbExplorer", 0, e.getMessage(), e );
        shell.getDisplay().asyncExec(new Runnable() {
        	public void run() {
        		ErrorDialog.openError( shell, title, message, status );
        	}
        });
        e.printStackTrace();
    }
    
    /**
     * Display an error popup.
     * 
     * @param e         The exception describing the problem, cannot be null
     * @param shell     The top-level shell
     * @param title     Title of the popup window
     * @param message   Text of the error message
     */
    public static void errorMessage(final Shell shell, 
                              final String title,
                              final String message ) {
        shell.getDisplay().asyncExec(new Runnable() {
        	public void run() {
        		MessageDialog.openError( shell, title, message);
        	}
        });
    }
}