/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.tree.helpers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmDefinitionList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.DefaultMemberList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultCodeList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultMemberList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class FaultFamilyHelper implements ThreeColumnDomainObjectHelper 
{
	private static final int ONE_THOUSAND_INT = 1000;
	private static final double ONE_THOUSAND_DOUBLE = 1000.0;
	@SuppressWarnings("unused")
	private Configuration configuration;
	private FaultFamily faultFamily;
	private static Map<Integer, FaultFamily> faultFamiliesMap = new HashMap<Integer, FaultFamily>();
	
	public FaultFamilyHelper(FaultFamily family, Configuration configuration) 
	{
		this.faultFamily = FaultFamilyHelper.findFaultFamily(family);
		this.configuration = configuration;
	}

	private void checkHydration()
	{
		this.faultFamily = FaultFamilyHelper.findFaultFamily(faultFamily);
	}
	
	@Override
	public boolean hasChildren() {
		return true;
	}
	
	@Override
	public Object[] getChildren() 
	{	
		checkHydration();
		
		FaultCodeList faultCodeList = new FaultCodeList(faultFamily);
		FaultMemberList faultMemberList = new FaultMemberList(faultFamily);
		DefaultMemberList defaultMemberList = new DefaultMemberList(faultFamily);
		
		List<AlarmDefinition> alarmDefs;
		AlarmDefinitionList[] adLists = null;
		try {
			alarmDefs = AlarmConversationUtils.getInstance().findAlarmDefinitionsWithReductionLinksForFaultFamily(faultFamily);
			
			AlarmDefinition[] alarmDefsArray = alarmDefs.toArray(new AlarmDefinition[alarmDefs.size()]);
			Arrays.sort(alarmDefsArray, AlarmDefinitionHelper.getComparator());

			// TODO: some of this code is no longer needed; there was once a time when we had large numbers
			// of alarm definitions, this is no longer true, so this stuff about pagination-like behavior should be removed.
			int numLists = (int) Math.ceil(alarmDefs.size() / ONE_THOUSAND_DOUBLE);
			adLists = new AlarmDefinitionList[numLists];
			try 
			{
				for(int i = 0; i < alarmDefsArray.length; i = i + ONE_THOUSAND_INT)
				{
					List<AlarmDefinition> currentList = new ArrayList<AlarmDefinition>();

					int start = i;
					int finish = (i + ONE_THOUSAND_INT) <= alarmDefs.size() ? i + ONE_THOUSAND_INT : alarmDefs.size(); 

					for(int j = start; j < finish; j++) {
						currentList.add(alarmDefsArray[j]);
					}
					adLists[i/ONE_THOUSAND_INT] = new AlarmDefinitionList(currentList, i, i + (ONE_THOUSAND_INT - 1), faultFamily);
				}
			} 
			catch(Throwable th) 
			{
				th.printStackTrace();
			}
			Object[] retVal = new Object[adLists.length];
			for(int i = 0; i < adLists.length; i++) {
				retVal[i] = adLists[i];
			}

		} catch (Exception e) {
			alarmDefs = null;
		}
		
		Object[] retVal = null;
		if(adLists != null) {
			retVal = new Object[3 + adLists.length];
			retVal[0] = faultMemberList;
			retVal[1] = defaultMemberList;
			retVal[2] = faultCodeList;
			for(int i = 0; i < adLists.length; i++)
			{
				retVal[i + 3] = adLists[i];
			}
		} else {
			retVal = new Object[3];
			retVal[0] = faultMemberList;
			retVal[1] = defaultMemberList;
			retVal[2] = faultCodeList;
		}

		return retVal;
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/faultfamily.png");
	}

	@Override
	public String getFirstColumnText() {
		return faultFamily.getFamilyName();
	}

	@Override
	public Font getFont() {
		return null;
	}

	@Override
	public Color getForeground() {
		return null;
	}

	@Override
	public Image getSecondColumnImage() {
		return null;
	}

	@Override
	public String getSecondColumnText() {
		return null;
	}

	@Override
	public Image getThirdColumnImage() {
		return null;
	}

	@Override
	public String getThirdColumnText() {
		return null;
	}

	public static synchronized FaultFamily findFaultFamily(FaultFamily ff) 
	{
		FaultFamily retDef = null;
		retDef = FaultFamilyHelper.faultFamiliesMap.get(ff.getFaultFamilyId());
		if(null == retDef) {
			try {
				if(ff.getFaultFamilyId() == null) {
					throw new IllegalStateException("FaultFamily's id is null - shouldn't happen");
				}
				ff = AlarmConversationUtils.getInstance().findFaultFamilyById(ff.getFaultFamilyId());
				ff = AlarmConversationUtils.getInstance().hydrateFaultFamily(ff);
				
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not hydrate FaultFamily", e);
			}
			FaultFamilyHelper.faultFamiliesMap.put(ff.getFaultFamilyId(), ff);
			retDef = ff;
		}
		return retDef;
	}

	public static void clearCache() {
		faultFamiliesMap.clear();
	}
}
