/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.tree.helpers;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionLinkType;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AlarmDefinitionHelper implements ThreeColumnDomainObjectHelper 
{
	private AlarmDefinition alarmDefinition;
	@SuppressWarnings("unused")
	private Configuration configuration;
	private static Map<Integer, AlarmDefinition> alarmDefinitionsMap = new HashMap<Integer, AlarmDefinition>();
	
	public AlarmDefinitionHelper(AlarmDefinition definition, Configuration configuration) 
	{
		this.alarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(definition);
		this.configuration = configuration;
	}

	@Override
	public Object[] getChildren() 
	{
		Object[] retVal = null;

		List<ReductionLink> nodeReductions = getNodeReductionsForAlarmDefinition();
		if(alarmDefinition.getReductionThreshold() != null && nodeReductions.size() > 0) 
		{
			retVal = new Object[nodeReductions.size() + 1];
			retVal[0] = ReductionThresholdHelper.findReductionThreshold(alarmDefinition.getReductionThreshold());
			int i = 1;
			for(ReductionLink link : nodeReductions) 
			{
				retVal[i] = ReductionLinkHelper.findReductionLink(link);
				i++;
			}
		}
		else if(alarmDefinition.getReductionThreshold() == null && nodeReductions.size() > 0) 
		{
			retVal = new Object[nodeReductions.size()];
			int count = 0;
			for(ReductionLink rlink : nodeReductions) {
				retVal[count++] = ReductionLinkHelper.findReductionLink(rlink);
			}
		}
		else if(alarmDefinition.getReductionThreshold() != null && nodeReductions.size() == 0) {
			retVal = new Object[1];
			retVal[0] = ReductionThresholdHelper.findReductionThreshold(alarmDefinition.getReductionThreshold());
		}
		
		return retVal;
	}

	private List<ReductionLink> getNodeReductionsForAlarmDefinition() {
		List<ReductionLink> nodeReductions = new ArrayList<ReductionLink>();
		for(ReductionLink link : alarmDefinition.getReductionLinksForParentalarmdefid())
		{
			if(link.getType().equals(ReductionLinkType.NODE)) {
				ReductionLink rl = ReductionLinkHelper.findReductionLink(link);
				nodeReductions.add(rl);
			}
		}
		return nodeReductions;
	}

	@Override
	public Image getFirstColumnImage() {
		return RcpUtils.getImage("icons/alarm-definition.png");
	}

	public static String getNameText(AlarmDefinition alarmDef)
	{
		if(null == alarmDef) {
			return null;
		}
		String retVal = "<";
		retVal += alarmDef.getFaultFamily();
		retVal += ", ";
		retVal += alarmDef.getFaultMember();
		retVal += ", ";
		retVal += alarmDef.getFaultCode();
		retVal += ">";
		
		return retVal;
	}
	
	@Override
	public String getFirstColumnText() 
	{
		return AlarmDefinitionHelper.getNameText(alarmDefinition);
	}

	@Override
	public Font getFont() {
		return null;
	}

	@Override
	public Color getForeground() {
		return null;
	}

	@Override
	public Image getSecondColumnImage() {
		return null;
	}

	@Override
	public String getSecondColumnText() {
		return null;
	}

	@Override
	public Image getThirdColumnImage() {
		return null;
	}

	@Override
	public String getThirdColumnText() {
		return null;
	}
	
	public AlarmDefinition getAlarmDefinition() {
		return this.alarmDefinition;
	}

	@Override
	public boolean hasChildren() 
	{
		boolean retVal = false;
		
		retVal = (alarmDefinition.getReductionThreshold() != null || 
				  getNodeReductionsForAlarmDefinition().size() > 0) 
				  ? true : false;
		
		return retVal;
	}

	public static synchronized AlarmDefinition findAlarmDefinition(AlarmDefinition alarmDef) 
	{
		AlarmDefinition retDef = null;
		retDef = AlarmDefinitionHelper.alarmDefinitionsMap.get(alarmDef.getAlarmDefinitionId());
		if(null == retDef) {
			try {
				if(alarmDef.getAlarmDefinitionId() == null) {
					throw new IllegalStateException("Alarm def's id is null - shouldn't happen");
				}
				alarmDef = AlarmConversationUtils.getInstance().findAlarmDefinitionById(alarmDef.getAlarmDefinitionId());
				alarmDef = AlarmConversationUtils.getInstance().hydrateAlarmDefinition(alarmDef);
			} catch (Exception e) {
				e.printStackTrace();
				throw new RuntimeException("Could not hydrate alarm definition", e);
			}
			AlarmDefinitionHelper.alarmDefinitionsMap.put(alarmDef.getAlarmDefinitionId(), alarmDef);
			retDef = alarmDef;
		}
		return retDef;
	}

	public static void clearCache() {
		alarmDefinitionsMap.clear();
	}
	
	public static AlarmDefinitionComparator getComparator()
	{
		return new AlarmDefinitionComparator();
	}
	
	private static class AlarmDefinitionComparator implements Comparator<AlarmDefinition>
	{
		@Override
		public int compare(AlarmDefinition o1, AlarmDefinition o2)
		{
			String s1 = AlarmDefinitionHelper.getNameText(o1);
			String s2 = AlarmDefinitionHelper.getNameText(o2);
			return s1.compareTo(s2);
		}
	}
}
