/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.editors.inputs.ReductionThresholdEditorInput;
import alma.obops.tmcdb.alarms.ui.widgets.ReductionThresholdAttributesComposite;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class ReductionThresholdEditor extends EditorPart implements
		DirtyListener, IModelChangePublisher 
{
	public static final String ID = "reductionthreshold.editor";
	private ReductionThresholdAttributesComposite downcastControl;
	private Integer originalValue;
	private ReductionThreshold reductionThreshold;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
			
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	private void rollbackEdits() 
	{
		this.reductionThreshold.setValue(originalValue);
	}

	private void commitEdits() {
		this.originalValue = downcastControl.getThreshold().getValue();
	}

	private void applyChangesAndSave() 
	{
		Integer newValue = downcastControl.getThreshold().getValue();
		if(!originalValue.equals(newValue)) 
		{
			shouldNotifyListeners = true;
			this.setPartName(newValue.toString());
		} 
		else {
			shouldNotifyListeners = false;
		}
		this.reductionThreshold.setValue(newValue);

		try {
			AlarmConversationUtils.getInstance().saveOrUpdateReductionThreshold(reductionThreshold, true);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		}
		
		this.downcastControl.setThreshold(this.reductionThreshold);
		this.downcastControl.setDirty(false);
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		ReductionThresholdEditorInput editorInput = (ReductionThresholdEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		reductionThreshold = editorInput.getReductionThreshold();
		originalValue = reductionThreshold.getValue();
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite contentComp = new Composite(sc1, SWT.NONE);
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		layout2.makeColumnsEqualWidth = true;
		contentComp.setLayout(layout2);

		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		downcastControl = new ReductionThresholdAttributesComposite(contentComp, SWT.NONE, reductionThreshold, null, this);
		downcastControl.setLayoutData(gridData);
		
		sc1.setContent(contentComp);
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
}
