/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.ReductionLink;
import alma.obops.tmcdb.alarms.ui.editors.inputs.ReductionLinkEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionLinkHelper;
import alma.obops.tmcdb.alarms.ui.widgets.AlarmDefinitionSelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.providers.AlarmDefinitionSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class ReductionLinkEditor extends EditorPart implements DirtyListener, IModelChangePublisher 
{
	public static final String ID = "nodereductionlink.editor";
	private ReductionLink reductionLink;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	private Configuration configuration;
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
    private AlarmDefinition originalChildAlarmDefinition;
    private AlarmDefinition newChildAlarmDefinition;
    private Text reductionLinkText;
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	private void commitEdits() {
		this.originalChildAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByChildalarmdefid());
		setDirty(false);
	}
	
	private void rollbackEdits() 
	{
		reductionLink.setAlarmDefinitionByChildalarmdefid(originalChildAlarmDefinition);
		newChildAlarmDefinition.getReductionLinksForChildalarmdefid().remove(reductionLink);
		originalChildAlarmDefinition.getReductionLinksForChildalarmdefid().add(reductionLink);
		try 
		{
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalChildAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newChildAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), "Problem rolling back changes", e.getMessage());
			e.printStackTrace();
		}
	}

	private void applyChangesAndSave() 
	{
		if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition)) 
		{
			shouldNotifyListeners = true;
			this.setPartName(AlarmDefinitionHelper.getNameText(newChildAlarmDefinition));
		} else {
			shouldNotifyListeners = false;
		}
		
		this.reductionLink.setAlarmDefinitionByChildalarmdefid(newChildAlarmDefinition);
		newChildAlarmDefinition.addReductionLinkToReductionLinksForChildalarmdefid(reductionLink);
		originalChildAlarmDefinition.getReductionLinksForChildalarmdefid().remove(reductionLink);
		
		try 
		{
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalChildAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newChildAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, "ReductionLink already exists: <Parent, Child> pairing must be unique");
			rollbackEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		ReductionLinkEditorInput editorInput = (ReductionLinkEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		reductionLink = editorInput.getReductionLink();
		this.originalChildAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByChildalarmdefid());
		this.configuration = reductionLink.getConfiguration();
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		Composite comp = new Composite(parent, SWT.NONE);
		GridLayout glayout = new GridLayout();
		glayout.numColumns = 2;
		glayout.makeColumnsEqualWidth = false;
		comp.setLayout(glayout);
		
		GridData gdata = new GridData();
		gdata.horizontalAlignment = SWT.BEGINNING;
		gdata.horizontalSpan = 1;
		gdata.verticalAlignment = SWT.BEGINNING;
		gdata.grabExcessHorizontalSpace = false;
		gdata.grabExcessVerticalSpace = false;
		
		Label fcLabel = new Label(comp, SWT.NONE);
		fcLabel.setText("Child Alarm Definition");
		fcLabel.setLayoutData(gdata);

		reductionLinkText = new Text(comp, SWT.BORDER | SWT.SINGLE);
		reductionLinkText.setText( reductionLink.getAlarmDefinitionByChildalarmdefid() == null ? 
				"" : AlarmDefinitionHelper.getNameText(reductionLink.getAlarmDefinitionByChildalarmdefid()));
		reductionLinkText.setEditable(false);
		gdata = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gdata.minimumWidth = 300;
		reductionLinkText.setLayoutData(gdata);
		
		Button button = new Button(comp, SWT.NONE);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionListener() {

			@Override
			public void widgetDefaultSelected(SelectionEvent evt) {
				widgetSelected(evt);
			}

			@Override
			public void widgetSelected(SelectionEvent evt) {
				AlarmDefinitionSelectionDialog selectionDialog = 
					new AlarmDefinitionSelectionDialog(getSite().getWorkbenchWindow(), new AlarmDefinitionSelectionDialogLabelProvider(), configuration, "Choose child alarm");
				selectionDialog.open();
				Object alarmDefs[] = selectionDialog.getResult();
				if( alarmDefs != null && alarmDefs.length == 1 ) {
					reductionLinkText.setText( AlarmDefinitionHelper.getNameText(((AlarmDefinition)alarmDefs[0])));
					newChildAlarmDefinition = (AlarmDefinition) alarmDefs[0];
				}
				setDirty(true);
			}
		});
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
		
		this.reductionLink = ReductionLinkHelper.findReductionLink(this.reductionLink);
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
		this.reductionLink = ReductionLinkHelper.findReductionLink(this.reductionLink);
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
}
