/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.obops.tmcdb.alarms.ui.editors.inputs.ReductionLinkEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionLinkHelper;
import alma.obops.tmcdb.alarms.ui.widgets.AlarmDefinitionComposite;
import alma.obops.tmcdb.alarms.ui.widgets.FaultCodeSelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.FaultFamilySelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.FaultMemberSelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.providers.FaultCodeSelectionDialogLabelProvider;
import alma.obops.tmcdb.alarms.ui.widgets.providers.FaultFamilySelectionDialogLabelProvider;
import alma.obops.tmcdb.alarms.ui.widgets.providers.FaultMemberSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class NodeReductionLinkEditor extends EditorPart implements DirtyListener, IModelChangePublisher 
{
	protected static final String CHANGES_NOT_SAVED = "Changes not saved";
	public static final String ID = "nodereductionlink.editor";
	protected AlarmDefinitionComposite downcastControl;
	protected boolean shouldNotifyListeners;
	protected boolean dirty;
	protected List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	protected ReductionLink reductionLink;
	protected Configuration configuration;
	protected AlarmDefinition originalChildAlarmDefinition, newChildAlarmDefinition;
	protected AlarmDefinition originalParentAlarmDefinition, newParentAlarmDefinition;
	protected String newParentFaultFamily;
	protected String newParentFaultMember;
	protected String newParentFaultCode;
	protected Button ffButton, fmButton, fcButton;
	protected Composite parentAlarmDefComposite;
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}

		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	@Override
	public void setFocus() {
	}


	@Override
	public void doSaveAs() {
		// not supported; noop
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		ReductionLinkEditorInput editorInput = (ReductionLinkEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		reductionLink = ReductionLinkHelper.findReductionLink(editorInput.getReductionLink());
		this.originalChildAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByChildalarmdefid());
		this.originalParentAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByParentalarmdefid());
		this.configuration = reductionLink.getConfiguration();
		this.newParentFaultCode = this.originalParentAlarmDefinition.getFaultCode();
		this.newParentFaultFamily = this.originalParentAlarmDefinition.getFaultFamily();
		this.newParentFaultMember = this.originalParentAlarmDefinition.getFaultMember();
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite scrolledComposite = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		scrolledComposite.setLayout(sc1Layout);
		scrolledComposite.setExpandHorizontal(true);
		scrolledComposite.setExpandVertical(true);
        
		Composite contentComposite = new Composite(scrolledComposite, SWT.NONE);
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		contentComposite.setLayout(layout2);

		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		downcastControl = new AlarmDefinitionComposite(contentComposite, SWT.NONE, null, this, this.configuration);
		downcastControl.setLayoutData(gridData);
		downcastControl.setReductionLink(reductionLink);
		
		parentAlarmDefComposite = new Composite(contentComposite, SWT.NONE);
		gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		parentAlarmDefComposite.setLayoutData(gridData);
		
		GridLayout layout3 = new GridLayout();
		layout3.numColumns = 3;
		parentAlarmDefComposite.setLayout(layout3);
		
		createFaultFamilyControls(parentAlarmDefComposite);
		createFaultMemberControls(parentAlarmDefComposite);
		createFaultCodeControls(parentAlarmDefComposite);
		
		scrolledComposite.setContent(contentComposite);
	}
	
	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	
	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}

	protected void commitEdits() {
		this.originalChildAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByChildalarmdefid());
		this.originalParentAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByParentalarmdefid());
		setDirty(false);
	}

	protected void rollbackEdits() 
	{
		boolean shouldCommit = false;
		
		if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) && 
				   !this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition)) 
		{
			// both parent and child alarm defs have changed
			rollbackChildAlarmDef();
			rollbackParentAlarmDef();
			shouldCommit = true;
		} else if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) ) {
			// only child alarm def has changed
			rollbackChildAlarmDef();
			shouldCommit = true;
		} else if(!this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition) ) {
			// only parent alarm def has changed
			rollbackParentAlarmDef();
			shouldCommit = true;
		} else {
			// neither parent nor child alarm defs have changed
			shouldCommit = false;
		}

		if(shouldCommit)
		{
			try 
			{
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
			} catch (ConstraintViolationException e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, "ReductionLink already exists: <Parent, Child> pairing must be unique");
				rollbackEdits();
			} catch (Exception e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
				e.printStackTrace();
			}
		}
	}
	
	protected void rollbackParentAlarmDef() 
	{
		reductionLink.setAlarmDefinitionByParentalarmdefid(originalParentAlarmDefinition);
		newParentAlarmDefinition.getReductionLinksForParentalarmdefid().remove(reductionLink);
		originalParentAlarmDefinition.getReductionLinksForParentalarmdefid().add(reductionLink);
		try 
		{
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalParentAlarmDefinition, false);
			if(null != newParentAlarmDefinition.getAlarmDefinitionId()) {
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newParentAlarmDefinition, false);
			}
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), "Problem rolling back parent alarm def", e.getMessage());
			e.printStackTrace();
		}
	}
	
	protected void rollbackChildAlarmDef() 
	{
		reductionLink.setAlarmDefinitionByChildalarmdefid(originalChildAlarmDefinition);
		newChildAlarmDefinition.getReductionLinksForChildalarmdefid().remove(reductionLink);
		originalChildAlarmDefinition.getReductionLinksForChildalarmdefid().add(reductionLink);
		try 
		{
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalChildAlarmDefinition, false);
			if(null != newChildAlarmDefinition.getAlarmDefinitionId()) {
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newChildAlarmDefinition, false);
			}
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), "Problem rolling back child alarm def", e.getMessage());
			e.printStackTrace();
		}
	}
	
	protected AlarmDefinition getAlarmDefinition(String familyName, String memberName, String faultCodeStr) throws Exception 
	{
		AlarmDefinition retVal = null;
		
		retVal = AlarmConversationUtils.getInstance().findMatchingAlarmDefinition(familyName, memberName, faultCodeStr, configuration);
		if(null == retVal) {
			retVal = new AlarmDefinition();
			retVal.setConfiguration(configuration);
			retVal.setFaultFamily(familyName);
			retVal.setFaultMember(memberName);
			retVal.setFaultCode(faultCodeStr);
		}
		else {
			retVal = AlarmDefinitionHelper.findAlarmDefinition(retVal);
		}
		
		return retVal;
	}

	protected void applyChangesAndSave()
	{
		try {
			this.newChildAlarmDefinition = getAlarmDefinition(downcastControl.getFaultFamily(), downcastControl.getFaultMember(), 
					                       downcastControl.getFaultCode());
			this.newParentAlarmDefinition = getAlarmDefinition(newParentFaultFamily, newParentFaultMember, newParentFaultCode);
		} catch (Exception e1) {
			e1.printStackTrace();
			throw new RuntimeException("Error when trying to find matching alarm definitions");
		}
		
		if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) && 
		   !this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition)) 
		{
			// both parent and child alarm defs have changed
			updateChildAlarmDef();
			updateParentAlarmDef();
		} else if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) ) {
			// only child alarm def has changed
			updateChildAlarmDef();
		} else if(!this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition) ) {
			// only parent alarm def has changed
			updateParentAlarmDef();
		} else {
			// neither parent nor child alarm defs have changed
			shouldNotifyListeners = false;
		}

		if(shouldNotifyListeners)
		{
			try 
			{
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
			} catch (ConstraintViolationException e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, "ReductionLink already exists: <Parent, Child> pairing must be unique");
				rollbackEdits();
			} catch (Exception e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
				e.printStackTrace();
			}
		}
	}

	protected void updateChildAlarmDef() {
		shouldNotifyListeners = true;
		this.setPartName(AlarmDefinitionHelper.getNameText(newChildAlarmDefinition));
		this.reductionLink.setAlarmDefinitionByChildalarmdefid(newChildAlarmDefinition);
		newChildAlarmDefinition.addReductionLinkToReductionLinksForChildalarmdefid(reductionLink);
		originalChildAlarmDefinition.getReductionLinksForChildalarmdefid().remove(reductionLink);
		try {
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalChildAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newChildAlarmDefinition, false);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
	}
	
	protected void updateParentAlarmDef() {
		shouldNotifyListeners = true;
		this.reductionLink.setAlarmDefinitionByParentalarmdefid(newParentAlarmDefinition);
		newParentAlarmDefinition.addReductionLinkToReductionLinksForParentalarmdefid(reductionLink);
		originalParentAlarmDefinition.getReductionLinksForParentalarmdefid().remove(reductionLink);
		try {
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalParentAlarmDefinition, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newParentAlarmDefinition, false);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
	}
	
	protected void createFaultFamilyControls(Composite parent) 
	{
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.BEGINNING;
		gd.verticalAlignment = GridData.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		Label fflabel = new Label(parent, SWT.NONE);
		fflabel.setText("Parent FF:");
		fflabel.setLayoutData(gd);
		
		final Text ffText =  new Text(parent, SWT.BORDER); 
		ffText.setText(reductionLink.getAlarmDefinitionByParentalarmdefid().getFaultFamily());
		ffText.setEditable(false);
		gd = new GridData();
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = false;
		ffText.setLayoutData(gd);
		
		ffButton = new Button(parent, SWT.NONE);
		ffButton.setText("Browse...");
		gd = new GridData();
		gd.horizontalAlignment = SWT.BEGINNING;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		ffButton.setLayoutData(gd);
		
		ffButton.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent evt) {
				widgetSelected(evt);
			}

			@Override
			public void widgetSelected(SelectionEvent evt) {
				FaultFamilySelectionDialog selectionDialog = 
					new FaultFamilySelectionDialog(getSite().getWorkbenchWindow(), new FaultFamilySelectionDialogLabelProvider(), configuration, "Choose parent fault family");
				selectionDialog.open();
				Object faultFamilies[] = selectionDialog.getResult();
				if( faultFamilies != null && faultFamilies.length == 1 ) {
					ffText.setText( ((FaultFamily)faultFamilies[0]).getFamilyName());
					newParentFaultFamily = ((FaultFamily)faultFamilies[0]).getFamilyName();
				}
				setDirty(true);
			}
		});
	}
	
	protected void createFaultMemberControls(Composite parent) 
	{
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.BEGINNING;
		gd.verticalAlignment = GridData.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		Label label = new Label(parent, SWT.NONE);
		label.setText("Parent FM:");
		label.setLayoutData(gd);
		
		final Text text =  new Text(parent, SWT.BORDER); 
		text.setText(reductionLink.getAlarmDefinitionByParentalarmdefid().getFaultMember());
		text.setEditable(false);
		gd = new GridData();
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = false;
		text.setLayoutData(gd);
		
		fmButton = new Button(parent, SWT.NONE);
		fmButton.setText("Browse...");
		gd = new GridData();
		gd.horizontalAlignment = SWT.BEGINNING;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		fmButton.setLayoutData(gd);
		
		fmButton.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent evt) {
				widgetSelected(evt);
			}

			@Override
			public void widgetSelected(SelectionEvent evt) {
				FaultFamily newParentFF = getParentFF();
				FaultMemberSelectionDialog selectionDialog = 
					new FaultMemberSelectionDialog(getSite().getWorkbenchWindow(), new FaultMemberSelectionDialogLabelProvider(), newParentFF);
				selectionDialog.open();
				Object faultMembers[] = selectionDialog.getResult();
				if( faultMembers != null && faultMembers.length == 1 ) {
					text.setText( ((FaultMember)faultMembers[0]).getMemberName());
					newParentFaultMember = ((FaultMember)faultMembers[0]).getMemberName();
				}
				setDirty(true);
			}
		});
	}
	
	protected FaultFamily getParentFF() {
		List<FaultFamily> newParentFFs;
		try {
			newParentFFs = AlarmConversationUtils.getInstance().findFaultFamiliesByRegexp(newParentFaultFamily, configuration);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("Could not locate fault family");
		}
		if(newParentFFs.size() != 1) {
			throw new IllegalStateException("No fault family with the chosen name");
		}
		FaultFamily newParentFF = newParentFFs.get(0);
		return newParentFF;
	}
	
	protected void createFaultCodeControls(Composite parent) 
	{
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.BEGINNING;
		gd.verticalAlignment = GridData.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		Label label = new Label(parent, SWT.NONE);
		label.setText("Parent FC:");
		label.setLayoutData(gd);
		
		final Text text =  new Text(parent, SWT.BORDER); 
		text.setText(reductionLink.getAlarmDefinitionByParentalarmdefid().getFaultCode());
		text.setEditable(false);
		gd = new GridData();
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = false;
		text.setLayoutData(gd);
		
		fcButton = new Button(parent, SWT.NONE);
		fcButton.setText("Browse...");
		gd = new GridData();
		gd.horizontalAlignment = SWT.BEGINNING;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		fcButton.setLayoutData(gd);
		
		fcButton.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetDefaultSelected(SelectionEvent evt) {
				widgetSelected(evt);
			}

			@Override
			public void widgetSelected(SelectionEvent evt) {
				FaultFamily newParentFF = getParentFF();
				FaultCodeSelectionDialog selectionDialog = 
					new FaultCodeSelectionDialog(getSite().getWorkbenchWindow(), new FaultCodeSelectionDialogLabelProvider(), newParentFF);
				selectionDialog.open();
				Object faultCodes[] = selectionDialog.getResult();
				if( faultCodes != null && faultCodes.length == 1 ) {
					text.setText( ((FaultCode)faultCodes[0]).getCodeValue().toString());
					newParentFaultCode = ((FaultCode)faultCodes[0]).getCodeValue().toString();
				}
				setDirty(true);
			}
		});
	}
}
