/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.wizards.support.IntegerVerifyListener;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;

public class MultiplicityReductionLinkEditor extends NodeReductionLinkEditor
{
	@SuppressWarnings("hiding")
	public static final String ID = "multiplicityreductionlink.editor";
	
	private Integer originalReductionThreshold;
	private Text thresholdText;
		
	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		super.init(site, input);
		this.originalReductionThreshold = this.originalParentAlarmDefinition.getReductionThreshold().getValue();
	}
	
	@Override
	public void createPartControl(Composite parent) 
	{
		super.createPartControl(parent);
		createReductionThresholdControls();
	}
	
	protected void applyChangesAndSave()
	{
		if(null == this.newParentFaultCode || null == this.newParentFaultFamily || null == this.newParentFaultMember 
		   || null == this.downcastControl.getFaultCode() || null == this.downcastControl.getFaultFamily() 
		   || null == this.downcastControl.getFaultMember() || null == this.thresholdText.getText().trim() 
		   || 0 == this.thresholdText.getText().trim().length())
		{
			final IStatus status = new Status( IStatus.WARNING, "TmcdbExplorer", 0, "Complete all fields", null);
			ErrorDialog.openError(getSite().getShell(), "Incomplete information", "Please complete all fields", status);
			return;
		}
		
		boolean updateConfig = false;
		
		try 
		{
			this.newChildAlarmDefinition = getAlarmDefinition(downcastControl.getFaultFamily(), downcastControl.getFaultMember(), 
					                       downcastControl.getFaultCode());
			this.newParentAlarmDefinition = getAlarmDefinition(newParentFaultFamily, newParentFaultMember, newParentFaultCode);
			
			if(null == this.newParentAlarmDefinition.getReductionThreshold()) 
			{
				ReductionThreshold newThreshold = new ReductionThreshold();
				newThreshold.setAlarmDefinition(this.newParentAlarmDefinition);
				newThreshold.setConfiguration(configuration);
				newThreshold.setValue(Integer.parseInt(thresholdText.getText().trim()));
				configuration.addReductionThresholdToReductionThresholds(newThreshold);
				updateConfig = true;
				this.newParentAlarmDefinition.setReductionThreshold(newThreshold);
			} 
			else 
			{
				this.newParentAlarmDefinition.getReductionThreshold().setValue(Integer.parseInt(thresholdText.getText().trim()));
			}
		} 
		catch (Exception e1) 
		{
			e1.printStackTrace();
			throw new RuntimeException("Error when trying to find matching alarm definitions");
		}
		
		if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) && 
		   !this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition)) 
		{
			// both parent and child alarm defs have changed
			updateChildAlarmDef();
			updateParentAlarmDef();
		} else if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) ) {
			// only child alarm def has changed
			updateChildAlarmDef();
		} else if(!this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition) ) {
			// only parent alarm def has changed
			updateParentAlarmDef();
		} else {
			// neither parent nor child alarm defs have changed
			shouldNotifyListeners = false;
		}
		
		if(!this.originalReductionThreshold.equals(Integer.parseInt(this.thresholdText.getText().trim()))) {
			updateReductionThreshold();
		}

		if(shouldNotifyListeners)
		{
			try 
			{
				if(updateConfig) 
				{
					SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(configuration, false);
				}
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
			} catch (ConstraintViolationException e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, "ReductionLink already exists: <Parent, Child> pairing must be unique");
				rollbackEdits();
			} catch (Exception e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
				e.printStackTrace();
			}
		}
	}
	
	protected void commitEdits() {
		this.originalChildAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByChildalarmdefid());
		this.originalParentAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(reductionLink.getAlarmDefinitionByParentalarmdefid());
		this.originalReductionThreshold = reductionLink.getAlarmDefinitionByParentalarmdefid().getReductionThreshold().getValue();
		setDirty(false);
	}

	protected void rollbackEdits() 
	{
		boolean shouldCommit = false;
		
		if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) && 
				   !this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition)) 
		{
			// both parent and child alarm defs have changed
			rollbackChildAlarmDef();
			rollbackParentAlarmDef();
			shouldCommit = true;
		} else if(!this.reductionLink.getAlarmDefinitionByChildalarmdefid().equals(newChildAlarmDefinition) ) {
			// only child alarm def has changed
			rollbackChildAlarmDef();
			shouldCommit = true;
		} else if(!this.reductionLink.getAlarmDefinitionByParentalarmdefid().equals(newParentAlarmDefinition) ) {
			// only parent alarm def has changed
			rollbackParentAlarmDef();
			shouldCommit = true;
		} else {
			// neither parent nor child alarm defs have changed
			shouldCommit = false;
		}

		if(shouldCommit)
		{
			try 
			{
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(reductionLink, true);
			} catch (ConstraintViolationException e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, "ReductionLink already exists: <Parent, Child> pairing must be unique");
				rollbackEdits();
			} catch (Exception e) {
				GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
				e.printStackTrace();
			}
		}
	}
		
	private void createReductionThresholdControls() 
	{
		GridData gd = new GridData();
		gd.horizontalAlignment = GridData.BEGINNING;
		gd.verticalAlignment = GridData.CENTER;
		gd.grabExcessHorizontalSpace = false;
		gd.grabExcessVerticalSpace = false;
		Label label = new Label(parentAlarmDefComposite, SWT.NONE);
		label.setText("Reduction Threshold:");
		label.setLayoutData(gd);
		
		thresholdText =  new Text(parentAlarmDefComposite, SWT.BORDER); 
		thresholdText.setText(this.originalReductionThreshold.toString());
		thresholdText.setEditable(true);
		gd = new GridData();
		gd.horizontalAlignment = SWT.FILL;
		gd.verticalAlignment = SWT.CENTER;
		gd.grabExcessHorizontalSpace = true;
		gd.grabExcessVerticalSpace = false;
		thresholdText.setLayoutData(gd);
		thresholdText.addVerifyListener(new IntegerVerifyListener());
		thresholdText.addModifyListener(new ModifyListener() {
			@Override
			public void modifyText(ModifyEvent evt) {
				setDirty(true);
			}
		});
	}
	
	private void updateReductionThreshold() 
	{
		shouldNotifyListeners = true;
		
		try {
			AlarmConversationUtils.getInstance().saveOrUpdateReductionThreshold(newParentAlarmDefinition.getReductionThreshold(), false);
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
	}
	
	protected void rollbackParentAlarmDef() 
	{
		reductionLink.setAlarmDefinitionByParentalarmdefid(originalParentAlarmDefinition);
		newParentAlarmDefinition.getReductionLinksForParentalarmdefid().remove(reductionLink);
		originalParentAlarmDefinition.getReductionLinksForParentalarmdefid().add(reductionLink);
		try 
		{
			if(!this.originalReductionThreshold.equals(Integer.parseInt(this.thresholdText.getText().trim()))) {
				this.originalParentAlarmDefinition.getReductionThreshold().setValue(originalReductionThreshold);
			}
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(originalParentAlarmDefinition, false);
			
			if(null != newParentAlarmDefinition.getAlarmDefinitionId()) {
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newParentAlarmDefinition, false);
			}
		} catch (Exception e) {
			GuiUtils.showErrorDialog(this.getSite().getShell(), "Problem rolling back parent alarm def", e.getMessage());
			e.printStackTrace();
		}
	}
}
