/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.editors.inputs.FaultMemberEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdb.alarms.ui.widgets.FaultMemberAttributesComposite;
import alma.obops.tmcdb.alarms.ui.widgets.LocationSelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.providers.LocationSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class FaultMemberEditor extends EditorPart implements DirtyListener, IModelChangePublisher 
{
	public static final String ID = "faultmember.editor";
	private FaultMemberAttributesComposite downcastControl;
	private LocationSelectionDialog locationSelectionDialog;
	private String originalName;
	private Location originalLocation;
	private FaultMember faultMember;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
			
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	private void rollbackEdits() 
	{
		this.faultMember.setMemberName(originalName);
		originalLocation = LocationHelper.findLocation(faultMember.getLocation());
		this.faultMember.setLocation(originalLocation);
		Location[] selections = new Location[] { faultMember.getLocation() };
		locationSelectionDialog.setInitialSelections(selections);
	}

	private void commitEdits() {
		this.originalName = downcastControl.getMemberName();
		this.originalLocation = LocationHelper.findLocation(getSelectedLocation());
	}

	private void applyChangesAndSave() 
	{
		String newFaultMemberName = downcastControl.getMemberName();
		if(!this.faultMember.getMemberName().equals(newFaultMemberName)) 
		{
			shouldNotifyListeners = true;
			this.setPartName(newFaultMemberName);
		} 
		else if((getSelectedLocation() != null && null == originalLocation ) ||
				(!getSelectedLocation().getMnemonic().equals(originalLocation.getMnemonic())) )
		{
			shouldNotifyListeners = true;			
		}
		else {
			shouldNotifyListeners = false;
		}
		this.faultMember.setMemberName(newFaultMemberName);
		this.faultMember.setLocation(getSelectedLocation());

		try {
			AlarmConversationUtils.getInstance().saveOrUpdateFaultMember(faultMember, true);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "Fault Member already exists: name must be unique within alarm category");
			faultMember.setMemberName(originalName);
			rollbackEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		}
		
		this.downcastControl.setFaultMember(this.faultMember);
		this.downcastControl.setDirty(false);
	}

	private Location getSelectedLocation() 
	{
		Object[] selectedObjs = (this.locationSelectionDialog != null) ? this.locationSelectionDialog.getResult() : null;
		Location retVal = (null != selectedObjs && selectedObjs.length > 0) ? (Location)selectedObjs[0] : faultMember.getLocation();
		retVal = LocationHelper.findLocation(retVal);
		return retVal;
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		FaultMemberEditorInput editorInput = (FaultMemberEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		faultMember = editorInput.getFaultMember();
		originalName = faultMember.getMemberName();
		originalLocation = faultMember.getLocation();
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{	
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite contentComp = new Composite(sc1, SWT.NONE);
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		layout2.makeColumnsEqualWidth = true;
		contentComp.setLayout(layout2);

		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		downcastControl = new FaultMemberAttributesComposite(contentComp, SWT.NONE, faultMember, null, this);
		downcastControl.setLayoutData(gridData);

		Composite locationComposite = new Composite(contentComp, SWT.NONE);
		GridLayout layout3 = new GridLayout();
		layout3.numColumns = 3;
		layout3.makeColumnsEqualWidth = false;
		locationComposite.setLayout(layout3);
		
		/* Location */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label locationLabel = new Label(locationComposite, SWT.NONE);
		locationLabel.setText("Location");
		locationLabel.setLayoutData(gd);

		final Text locationText = new Text(locationComposite, SWT.BORDER | SWT.SINGLE);
		locationText.setText( faultMember.getLocation() == null ? "" : faultMember.getLocation().getMnemonic() );
		locationText.setEditable(false);
		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.minimumWidth = 300;
		locationText.setLayoutData(gd);
		

		gd = new GridData(SWT.LEFT, SWT.BEGINNING, false, false);
		Button browseLocationsButton = new Button(locationComposite, SWT.PUSH);
		browseLocationsButton.setText("Browse...");
		browseLocationsButton.setLayoutData(gd);
	
		// Setup the browser button and logic
		browseLocationsButton.addSelectionListener(new SelectionListener() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				locationSelectionDialog = new LocationSelectionDialog(getSite().getShell(), new LocationSelectionDialogLabelProvider());
				Location[] selections = new Location[] { LocationHelper.findLocation(faultMember.getLocation()) };
				locationSelectionDialog.setInitialSelections(selections);
				locationSelectionDialog.open();
				Object locations[] = locationSelectionDialog.getResult();
				if( locations != null && locations.length == 1 ) {
					locationText.setText( ((Location)locations[0]).getMnemonic() );
				}
				setDirty(true);
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		
		sc1.setContent(contentComp);
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
		
		this.faultMember = FaultMemberHelper.findFaultMember(this.faultMember);
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
		this.faultMember = FaultMemberHelper.findFaultMember(this.faultMember);
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
	
}
