/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.Contact;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdb.alarms.ui.editors.inputs.FaultFamilyEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ContactHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.widgets.ContactSelectionDialog;
import alma.obops.tmcdb.alarms.ui.widgets.FaultFamilyAttributesComposite;
import alma.obops.tmcdb.alarms.ui.widgets.providers.ContactSelectionDialogLabelProvider;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class FaultFamilyEditor extends EditorPart implements DirtyListener, IModelChangePublisher 
{
	public static final String ID = "faultfamily.editor";
	private FaultFamilyAttributesComposite downcastControl;
	private ContactSelectionDialog contactSelectionDialog;
	private FaultFamily faultFamily;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
    private Text contactText;
	private String originalName;
	private String originalAlarmSource;
	private Contact originalContact;
	private String originalHelpUrl;
	
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			rollbackEdits();
			e.printStackTrace();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}

	private void applyChangesAndSave() 
	{
		String newFaultFamilyName = downcastControl.getFamilyName();
		if(!this.faultFamily.getFamilyName().equals(newFaultFamilyName)) 
		{
			shouldNotifyListeners = true;
			this.setPartName(newFaultFamilyName);
		} 
		else if(!this.originalContact.getContactId().equals(getSelectedContact().getContactId()) || 
				!this.originalAlarmSource.equals(downcastControl.getAlarmSource()) ||
				!this.originalHelpUrl.equals(downcastControl.getHelpUrl()))
		{
			shouldNotifyListeners = true;
		}
		else {
			shouldNotifyListeners = false;
		}
		
		this.faultFamily.setFamilyName(newFaultFamilyName);
		faultFamily.setAlarmSource(downcastControl.getAlarmSource());
		faultFamily.setContact(getSelectedContact());
		faultFamily.setHelpURL(downcastControl.getHelpUrl());

		try {
			AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(faultFamily, true);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "Fault Family already exists: name must be unique within alarm category");
			rollbackEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			rollbackEdits();
			e.printStackTrace();
		}
		
		this.downcastControl.setFaultFamily(this.faultFamily);
		this.downcastControl.setDirty(false);
	}
	
	private void commitEdits() {
		originalName = faultFamily.getFamilyName();
		originalHelpUrl = faultFamily.getHelpURL();
		originalAlarmSource = faultFamily.getAlarmSource();
		originalContact = ContactHelper.findContact(faultFamily.getContact());
	}

	private void rollbackEdits() {
		faultFamily.setFamilyName(originalName);
		faultFamily.setHelpURL(originalHelpUrl);
		faultFamily.setAlarmSource(originalAlarmSource);
		originalContact = ContactHelper.findContact(originalContact);
		faultFamily.setContact(originalContact);
		this.downcastControl.setFaultFamily(faultFamily);
		contactText.setText(originalContact.getContactName());
		setPartName(originalName);
		setDirty(false);
	}

	private Contact getSelectedContact() 
	{
		Object[] selectedObjs = this.contactSelectionDialog.getResult();
		Contact retVal = (null != selectedObjs && selectedObjs.length > 0) ? (Contact)selectedObjs[0] : faultFamily.getContact();
		retVal = ContactHelper.findContact(retVal);
		return retVal;
	}

	@Override
	public void init(IEditorSite site, IEditorInput input) throws PartInitException 
	{
		FaultFamilyEditorInput editorInput = (FaultFamilyEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		faultFamily = editorInput.getFaultFamily();
		this.originalName = faultFamily.getFamilyName();
		this.originalAlarmSource = faultFamily.getAlarmSource();
		this.originalContact = faultFamily.getContact();
		this.originalHelpUrl = faultFamily.getHelpURL();
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite contentComp = new Composite(sc1, SWT.NONE);
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		layout2.makeColumnsEqualWidth = true;
		contentComp.setLayout(layout2);

		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		downcastControl = new FaultFamilyAttributesComposite(contentComp, SWT.NONE, faultFamily, null, this, null);
		downcastControl.setLayoutData(gridData);

		Composite contactComposite = new Composite(contentComp, SWT.NONE);
		GridLayout layout3 = new GridLayout();
		layout3.numColumns = 3;
		layout3.makeColumnsEqualWidth = false;
		contactComposite.setLayout(layout3);
		
		/* Contact */
		GridData gd = new GridData(SWT.LEFT, SWT.CENTER, false, false);
		Label contactLabel = new Label(contactComposite, SWT.NONE);
		contactLabel.setText("Contact");
		contactLabel.setLayoutData(gd);

		contactText = new Text(contactComposite, SWT.BORDER | SWT.SINGLE);
		contactText.setText( faultFamily.getContact() == null ? "" : faultFamily.getContact().getContactName() );
		contactText.setEditable(false);
		gd = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd.minimumWidth = 300;
		int offset = 38; // TODO: find a better way to lay this out w/o "magic" numbers...
		gd.horizontalIndent = offset;
		contactText.setLayoutData(gd);

		gd = new GridData(SWT.LEFT, SWT.BEGINNING, false, false);
		Button browseContactsButton = new Button(contactComposite, SWT.PUSH);
		browseContactsButton.setText("Browse...");
		browseContactsButton.setLayoutData(gd);

		// Setup the browser button and logic
		contactSelectionDialog = new ContactSelectionDialog(getSite().getShell(), new ContactSelectionDialogLabelProvider());
		browseContactsButton.addSelectionListener(new SelectionListener() 
		{
			public void widgetSelected(SelectionEvent e) 
			{
				contactSelectionDialog.open();
				Object contacts[] = contactSelectionDialog.getResult();
				if( contacts != null && contacts.length == 1 ) {
					contactText.setText( ((Contact)contacts[0]).getContactName() );
				}
				setDirty(true);
			}
			public void widgetDefaultSelected(SelectionEvent e) {}
		});

		sc1.setContent(contentComp);
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
		
		this.faultFamily = FaultFamilyHelper.findFaultFamily(this.faultFamily);
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
		this.faultFamily = FaultFamilyHelper.findFaultFamily(this.faultFamily);
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
	
}
