/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IWorkbenchPartConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.FaultCode;
import alma.obops.tmcdb.alarms.ui.editors.inputs.FaultCodeEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultCodeHelper;
import alma.obops.tmcdb.alarms.ui.widgets.FaultCodeAttributesComposite;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class FaultCodeEditor extends EditorPart implements DirtyListener, IModelChangePublisher 
{
	public static final String ID = "faultcode.editor";
	private FaultCodeAttributesComposite downcastControl;
	private Integer originalValue;
	private Integer originalPriority;
	private String originalProblemDescription;
	private String originalAction;
	private String originalCause;
	private String originalConsequence;
	private boolean originalIsInstant;
	
	private FaultCode faultCode;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}

		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	private void commitEdits() {
		this.originalValue = downcastControl.getCodeValue();
		this.originalAction = downcastControl.getAction();
		this.originalCause = downcastControl.getCause();
		this.originalConsequence = downcastControl.getConsequence();
		this.originalIsInstant = downcastControl.getInstant();
		this.originalPriority = downcastControl.getPriority();
		this.originalProblemDescription = downcastControl.getProblemDescription();
	}

	private void rollbackEdits() {
		faultCode.setCodeValue(originalValue);
		faultCode.setAction(originalAction);
		faultCode.setCause(originalCause);
		faultCode.setConsequence(originalConsequence);
		faultCode.setIsInstant(originalIsInstant);
		faultCode.setPriority(originalPriority);
		faultCode.setProblemDescription(originalProblemDescription);
		this.setPartName(originalValue.toString());
	}
	
	private void applyChangesAndSave() 
	{
		Integer newFaultCodeValue = downcastControl.getCodeValue();
		if(!this.faultCode.getCodeValue().equals(newFaultCodeValue)) {
			shouldNotifyListeners = true;
			this.setPartName(newFaultCodeValue.toString());
		} else {
			shouldNotifyListeners = false;
		}
		this.faultCode.setCodeValue(newFaultCodeValue);
		this.faultCode.setAction(downcastControl.getAction());
		this.faultCode.setCause(downcastControl.getCause());
		this.faultCode.setConsequence(downcastControl.getConsequence());
		this.faultCode.setIsInstant(downcastControl.getInstant());
		this.faultCode.setPriority(downcastControl.getPriority());
		this.faultCode.setProblemDescription(downcastControl.getProblemDescription());

		try {
			AlarmConversationUtils.getInstance().saveOrUpdateFaultCode(faultCode, true);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "Fault Code already exists: name must be unique within alarm category");
			rollbackEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
			rollbackEdits();
		}
		
		this.downcastControl.setFaultCode(this.faultCode);
		this.downcastControl.setDirty(false);
	}
	
	@Override
	public void setInput( IEditorInput input ) 
	{
		super.setInput(input);
		FaultCodeEditorInput fcEdInput = ((FaultCodeEditorInput)input);
		FaultCode fc = (fcEdInput).getFaultCode();
		this.modelChangeListeners.clear();
		this.addModelChangeListener(fcEdInput.getModelChangeListener());
		this.faultCode = fc;
		this.originalValue = fc.getCodeValue();
		this.originalAction = fc.getAction();
		this.originalCause = fc.getCause();
		this.originalConsequence = fc.getConsequence();
		this.originalIsInstant = fc.getIsInstant();
		this.originalPriority = fc.getPriority();
		this.originalProblemDescription = fc.getProblemDescription();
		
		if(null != downcastControl) 
		{
			downcastControl.setFaultCode(fc);
		}
		
		firePropertyChange(IWorkbenchPartConstants.PROP_INPUT);
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		setInput(input);
		setSite(site);
		setPartName(input.getName());
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite contentComp = new Composite(sc1, SWT.NONE);
		GridLayout layout2 = new GridLayout();
		layout2.numColumns = 1;
		layout2.makeColumnsEqualWidth = true;
		contentComp.setLayout(layout2);

		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false, 1, 1);
		downcastControl = new FaultCodeAttributesComposite(contentComp, SWT.NONE, faultCode, null, this);
		downcastControl.setLayoutData(gridData);

		sc1.setContent(contentComp);
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
		
		this.faultCode = FaultCodeHelper.findFaultCode(this.faultCode);
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
		this.faultCode = FaultCodeHelper.findFaultCode(this.faultCode);
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
}
