/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.Configuration;
import alma.obops.tmcdb.alarms.ui.editors.inputs.AlarmCategoryEditorInput;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmCategoryHelper;
import alma.obops.tmcdb.alarms.ui.widgets.AlarmCategoryAttributesComposite;
import alma.obops.tmcdbgui.domain.IModelChangeListener;
import alma.obops.tmcdbgui.domain.IModelChangePublisher;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.widgets.support.DirtyListener;

public class AlarmCategoryEditor extends EditorPart implements DirtyListener, IModelChangePublisher
{
	public static final String ID = "alarmcategory.editor";
	private AlarmCategoryAttributesComposite downcastControl;
	private AlarmCategory alarmCategory;
	private boolean dirty;
	private boolean shouldNotifyListeners;
    private List<IModelChangeListener> modelChangeListeners = new ArrayList<IModelChangeListener>();
	private Configuration configuration;
    private static final String CHANGES_NOT_SAVED = "Changes not saved";
	private String originalName;
    private String originalDescription;
	private String originalPath;
	private boolean originalIsDefault;
	
	@Override
	public void doSave(IProgressMonitor monitor) 
	{
		try {
			this.getSite().getShell().setCursor(this.getSite().getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			applyChangesAndSave();
			commitEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		} 
		finally {
			this.getSite().getShell().setCursor(null);
		}
		
		if(shouldNotifyListeners) {
			this.modelChanged();
			this.shouldNotifyListeners = false;
		}
		setDirty(false);
	}
	
	private void commitEdits() {
		this.originalName = downcastControl.getCategoryName();
		this.originalDescription = downcastControl.getCategoryDescription();
		this.originalIsDefault = downcastControl.isDefault();
		this.originalPath = downcastControl.getCategoryPath();
	}
	
	private void rollbackEdits() 
	{
		alarmCategory.setAlarmCategoryName(originalName);
		alarmCategory.setDescription(originalDescription);
		alarmCategory.setPath(originalPath);
		alarmCategory.setIsDefault(originalIsDefault);
		setPartName(originalName);
		this.downcastControl.setAlarmCategory(this.alarmCategory);
		this.downcastControl.setDirty(false);
	}

	private void applyChangesAndSave() 
	{
		String newAlarmCategoryName = downcastControl.getCategoryName();
		if(!this.alarmCategory.getAlarmCategoryName().equals(newAlarmCategoryName)) {
			shouldNotifyListeners = true;
			this.setPartName(newAlarmCategoryName);
		} else {
			shouldNotifyListeners = false;
		}
		
		this.alarmCategory.setAlarmCategoryName(newAlarmCategoryName);
		alarmCategory.setDescription(downcastControl.getCategoryDescription());
		alarmCategory.setIsDefault(downcastControl.isDefault());
		alarmCategory.setPath(downcastControl.getCategoryPath());

		try {
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmCategory(alarmCategory, true);
		} catch (ConstraintViolationException e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, "AlarmCategory already exists: name must be unique within configuration");
			rollbackEdits();
		} catch (Exception e) {
			GuiUtils.showErrorDialog(downcastControl.getShell(), CHANGES_NOT_SAVED, e.getMessage());
			e.printStackTrace();
		}
		
		this.downcastControl.setAlarmCategory(this.alarmCategory);
		this.downcastControl.setDirty(false);
	}

	@Override
	public void init(IEditorSite site, IEditorInput input)
			throws PartInitException 
	{
		AlarmCategoryEditorInput editorInput = (AlarmCategoryEditorInput)input;
		this.addModelChangeListener(editorInput.getModelChangeListener());
		setInput(input);
		setSite(site);
		setPartName(editorInput.getName());
		alarmCategory = editorInput.getAlarmCategory();
		configuration = editorInput.getConfiguration();
	}

	@Override
	public void doSaveAs() {
		// noop - not allowed
	}

	@Override
	public boolean isDirty() {
		return dirty;
	}

	@Override
	public boolean isSaveAsAllowed() {
		// not allowed
		return false;
	}

	@Override
	public void createPartControl(Composite parent) 
	{
		parent.setLayout(new FillLayout());
		ScrolledComposite sc1 = new ScrolledComposite(parent,SWT.H_SCROLL |
				SWT.V_SCROLL | SWT.BORDER);
		FillLayout sc1Layout = new FillLayout(org.eclipse.swt.SWT.HORIZONTAL);
		sc1.setLayout(sc1Layout);
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
        
		Composite comp = new Composite(sc1, SWT.NONE);
		comp.setLayout(new FillLayout());
		downcastControl = new AlarmCategoryAttributesComposite(comp, SWT.NONE, alarmCategory, null, this, configuration);
		sc1.setContent(comp);
		this.originalName = alarmCategory.getAlarmCategoryName();
		this.originalDescription = alarmCategory.getDescription();
		this.originalPath = alarmCategory.getPath();
		this.originalIsDefault = alarmCategory.getIsDefault();
	}

	@Override
	public void setFocus() {
	}

	@Override
	public void setDirty(boolean dirty) {
		this.dirty = dirty;
		firePropertyChange(PROP_DIRTY);
	}
	

	@Override
	public void addModelChangeListener(IModelChangeListener listener) {
		if(null != listener)
		{
			this.modelChangeListeners.add(listener);
		}
	}

	@Override
	public void modelChanged() 
	{
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.internalModelChange();
		}
		this.alarmCategory = AlarmCategoryHelper.findAlarmCategory(this.alarmCategory);
	}

	@Override
	public void modelShouldBeReloaded() {
		for(IModelChangeListener listener: modelChangeListeners )
		{
			listener.externalModelChange();
		}
		this.alarmCategory = AlarmCategoryHelper.findAlarmCategory(this.alarmCategory);
	}

	@Override
	public void removeModelChangeListener(IModelChangeListener listener) {
		this.modelChangeListeners.remove(listener);
	}
	
}
