/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.delete;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class DeleteFaultMemberAction extends ConversationalAction 
{
	private FaultMember member;
	private boolean cancelled;
	private final static String CONSTRAINT_VIOLATION = "This default member cannot be deleted due to DB constraints.";
	private static final String ID = "delete_defaultmember.action";

	public DeleteFaultMemberAction(IWorkbenchWindow win) 
	{
		this.window = win;
		setId(ID);
		setText("Delete fault member");
		setToolTipText("Deletes an existing fault member");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/faultmember-delete.png" ));
		this.window.getSelectionService().addSelectionListener(this);
	}

	@Override public void doPreConversational() 
	{
		cancelled = !(MessageDialog.openConfirm(this.window.getShell(), "Please confirm", "Are you sure?"));
	}
	
	@Override
	public void doConversational() 
	{
		if(cancelled) {
			return;
		}
		try {
			FaultFamily family = FaultFamilyHelper.findFaultFamily(member.getFaultFamily());
			Location location = LocationHelper.findLocation(member.getLocation());
			AlarmConversationUtils.getInstance().removeFaultMember(family, location, member);
		} 
		catch(ConstraintViolationException ex) 
		{
			MessageDialog.openWarning(this.window.getShell(), "Cannot delete fault member", CONSTRAINT_VIOLATION );
		}
		catch (Exception e) 
		{
			MessageDialog.openError(this.window.getShell(), "Cannot delete fault member", "This fault member cannot be deleted for unknown reasons." );
			e.printStackTrace();
		}
	}
	
	@Override
	public void doPostConversational() {
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() && selection.getFirstElement() instanceof FaultMember)	
			{
				setEnabled(true);
				this.member = FaultMemberHelper.findFaultMember((FaultMember) selection.getFirstElement());
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
}
