/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignReductionThresholdAttributes;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionThresholdHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddReductionThresholdWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddReductionThresholdAction extends ConversationalAction implements IAssignReductionThresholdAttributes
{
	private AlarmDefinition currentlySelectedAlarmDefinition;
	private Integer thresholdValue;
	private static final String ID = "add_reductionthreshold.action";
	private Configuration configuration;
	
	public AddReductionThresholdAction(IWorkbenchWindow window, Configuration config)
	{
		this.window = window;
		setId(ID);
		setText("Add reduction threshold");
		setToolTipText("Adds a new reduction threhsold");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/reduction-threshold-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.configuration = config;
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();

		ReductionThreshold newReductionThreshold = null;
		
		try {
			// Collect user input
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			currentlySelectedAlarmDefinition = AlarmDefinitionHelper.findAlarmDefinition(currentlySelectedAlarmDefinition);
			AddReductionThresholdWizard wizard = new AddReductionThresholdWizard( this );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newReductionThreshold = new ReductionThreshold();
			newReductionThreshold.setConfiguration(configuration);
			newReductionThreshold.setAlarmDefinition(currentlySelectedAlarmDefinition);
			newReductionThreshold.setValue(thresholdValue);
			this.currentlySelectedAlarmDefinition.setReductionThreshold(newReductionThreshold);
			
			AlarmConversationUtils.getInstance().saveOrUpdateReductionThreshold(newReductionThreshold, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(this.currentlySelectedAlarmDefinition, true);
			ReductionThresholdHelper.findReductionThreshold(newReductionThreshold);
		}
		catch( Exception e ) {
			rollbackChanges(newReductionThreshold, e);
		} finally {
			this.window.getShell().setCursor(null);
		}
	}

	private void rollbackChanges(ReductionThreshold newReductionThreshold, Exception e) 
	{
		String message = e.getMessage();
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		e.printStackTrace();
		
		if(null != newReductionThreshold)
		{
			currentlySelectedAlarmDefinition.setReductionThreshold(null);
			
			try {
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(this.currentlySelectedAlarmDefinition, true);
			} catch (Exception e1) {
				GuiUtils.showErrorDialog(window.getShell(), "Problems rolling back transaction", e1.getMessage());
				e1.printStackTrace();
			}
		}
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.refreshTreeAndMaintainSelection();
		view.clearCaches();
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   ((selection.getFirstElement() instanceof AlarmDefinition)) )	
			{
				this.currentlySelectedAlarmDefinition = (AlarmDefinition) selection.getFirstElement();
				if(this.currentlySelectedAlarmDefinition.getReductionThreshold() == null)
				{
					setEnabled(true);
				} else {
					setEnabled(false);
				}
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void setReductionThresholdValue(Integer val) {
		this.thresholdValue = val;
	}

	public void setConfiguration(Configuration swConfiguration) {
		this.configuration = swConfiguration;
	}
}
