/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionLinkAction;
import alma.acs.tmcdb.ReductionLinkType;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignNodeReductionLinkAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewReductionLinkListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewReductionLinkPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionLinkHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddNodeReductionLinkWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;

public class AddNodeReductionLinkAction extends ConversationalAction implements INewReductionLinkPublisher, IAssignNodeReductionLinkAttributes
{
	private static final String ID = "add_node_reduction_link.action";
	
	private Collection<INewReductionLinkListener> newReductionLinkListeners = new ArrayList<INewReductionLinkListener>();
	private String childFaultFamily;
	private String childFaultMember;
	private String childFaultCode;
	private FaultFamily parentFamily;
	private FaultMember parentMember;
	private FaultCode parentCode;
	private ReductionLink newReductionLink;
	private Configuration configuration;
	
	public AddNodeReductionLinkAction(IWorkbenchWindow window, INewReductionLinkListener listener, Configuration config)
	{
		this.window = window;
		setId(ID);
		setText("New node reduction");
		setToolTipText("Adds a new node reduction link");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/reductionlink-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.addNewReductionLinkListener(listener);
		this.configuration = config;
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();
		
		try {
			// Collect user input
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddNodeReductionLinkWizard wizard = new AddNodeReductionLinkWizard( window, this, configuration);
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newReductionLink = new ReductionLink();
			newReductionLink.setType(ReductionLinkType.NODE);
			newReductionLink.setAction(ReductionLinkAction.REMOVE);
			newReductionLink.setConfiguration(ConfigurationHelper.findConfiguration(configuration));
			
			AlarmDefinition parentDefinition = 
				getAlarmDefinition(parentFamily.getFamilyName(), parentMember.getMemberName(), parentCode.getCodeValue().toString(), configuration);
			AlarmDefinition childDefinition = 
				getAlarmDefinition(childFaultFamily, childFaultMember, childFaultCode, configuration);
			
			newReductionLink.setAlarmDefinitionByParentalarmdefid(parentDefinition);
			newReductionLink.setAlarmDefinitionByChildalarmdefid(childDefinition);
			newReductionLink.getConfiguration().addReductionLinkToReductionLinks(newReductionLink);
			parentDefinition.addReductionLinkToReductionLinksForParentalarmdefid(newReductionLink);
			childDefinition.addReductionLinkToReductionLinksForChildalarmdefid(newReductionLink);
			
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			try {
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(newReductionLink, false);
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(childDefinition, false);
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(parentDefinition, false);
				SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(configuration, true);
				newReductionLink = ReductionLinkHelper.findReductionLink(newReductionLink);
				this.notifyNewReductionLinkListeners(newReductionLink);
			} 
			catch( Exception e ) {
				rollbackChanges(newReductionLink, e);
			} 
		}
		catch(Exception ex) {
			ex.printStackTrace();
			throw new RuntimeException("Unexpected problem with wizard");
		} finally {
			shell.setCursor(null);
		}
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			setEnabled(true);
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void addNewReductionLinkListener(INewReductionLinkListener listener) {
		if(null != listener) {
			this.newReductionLinkListeners.add(listener);
		}
	}

	@Override
	public void notifyNewReductionLinkListeners(ReductionLink newReductionLink1) 
	{
		for(INewReductionLinkListener listener: newReductionLinkListeners)
		{
			listener.update(newReductionLink1);
		}
	}

	@Override
	public void removeNewReductionLinkListener(INewReductionLinkListener listener) {
		this.newReductionLinkListeners.remove(listener);
	}

	@Override
	public void setChildFaultCode(String faultCode) {
		this.childFaultCode = faultCode;
	}

	@Override
	public void setChildFaultFamily(String family) {
		this.childFaultFamily = family;
	}

	@Override
	public void setChildFaultMember(String faultMember) {
		this.childFaultMember = faultMember;
	}

	@Override
	public void setParentFaultCode(FaultCode code) {
		this.parentCode = code;
	}

	@Override
	public void setParentFaultFamily(FaultFamily family) {
		this.parentFamily = family;
	}

	@Override
	public void setParentFaultMember(FaultMember member) {
		this.parentMember = member;
	}

	private void rollbackChanges(ReductionLink newReductionLink1, Exception e) 
	{
		String message = e.getMessage();
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newReductionLink1 && newReductionLink1.getConfiguration() != null)
		{
			Configuration config = configuration;
			config.getReductionLinks().remove(newReductionLink1);
			newReductionLink1.setConfiguration(null);
			try {
				AlarmDefinition parentDefinition = 
					getAlarmDefinition(parentFamily.getFamilyName(), parentMember.getMemberName(), parentCode.getCodeValue().toString(), configuration);
				AlarmDefinition childDefinition = 
					getAlarmDefinition(childFaultFamily, childFaultMember, childFaultCode, configuration);
				
				if(null != parentDefinition.getAlarmDefinitionId()) {
					parentDefinition.getReductionLinksForParentalarmdefid().remove(newReductionLink1);
					AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(parentDefinition, false);	
				}
				if(null != childDefinition.getAlarmDefinitionId()) {
					childDefinition.getReductionLinksForChildalarmdefid().remove(newReductionLink1);
					AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(childDefinition, false);	
				}
				SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
			} catch(Exception ex) {
				ex.printStackTrace();
				GuiUtils.showErrorDialog(window.getShell(), "Error rolling back transaction", "could not roll back cleanly");		
			}
		}
		e.printStackTrace();
	}

	private AlarmDefinition getAlarmDefinition(String familyName, String memberName, String faultCodeStr, Configuration config) throws Exception 
	{
		AlarmDefinition retVal = null;
		
		retVal = AlarmConversationUtils.getInstance().findMatchingAlarmDefinition(familyName, memberName, faultCodeStr, config);
		if(null == retVal) {
			retVal = new AlarmDefinition();
			retVal.setConfiguration(config);
			retVal.setFaultFamily(familyName);
			retVal.setFaultMember(memberName);
			retVal.setFaultCode(faultCodeStr);
		}
		else {
			retVal = AlarmDefinitionHelper.findAlarmDefinition(retVal);
		}
		
		return retVal;
	}
	
	public void setConfiguration(Configuration config) {
		this.configuration = config;
	}
}
