/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.ReductionLink;
import alma.acs.tmcdb.ReductionLinkAction;
import alma.acs.tmcdb.ReductionLinkType;
import alma.acs.tmcdb.ReductionThreshold;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignMultiplicityReductionLinkAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewReductionLinkListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewReductionLinkPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ReductionLinkHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddMultiplicityReductionLinkWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;

public class AddMultiplicityReductionLinkAction extends ConversationalAction
		implements INewReductionLinkPublisher, IAssignMultiplicityReductionLinkAttributes 
{
	private static final String ID = "add_multiplicity_reduction_link.action";
	private Collection<INewReductionLinkListener> newReductionLinkListeners = new ArrayList<INewReductionLinkListener>();
	private AlarmDefinition parentAlarmDefinition;
	private AlarmDefinition childAlarmDefinition;
	private String parentFaultFamily, parentFaultMember, parentFaultCode;
	private String childFaultFamily, childFaultMember, childFaultCode;
	private ReductionThreshold reductionThreshold;
	private Configuration configuration;
	
	public AddMultiplicityReductionLinkAction(IWorkbenchWindow window, INewReductionLinkListener listener, Configuration config)
	{
		this.window = window;
		setId(ID);
		setText("New multiplicity reduction");
		setToolTipText("Adds a new multiplicity reduction link");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/reductionlink-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.addNewReductionLinkListener(listener);
		this.configuration = config;
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();
		ReductionLink newReductionLink = null;
		
		try {
			// Collect user input
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddMultiplicityReductionLinkWizard wizard = new AddMultiplicityReductionLinkWizard( window, this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newReductionLink = new ReductionLink();
			newReductionLink.setType(ReductionLinkType.MULTIPLICITY);
			newReductionLink.setAction(ReductionLinkAction.CREATE);
			newReductionLink.setConfiguration(ConfigurationHelper.findConfiguration(configuration));
			
			parentAlarmDefinition = 
				getAlarmDefinition(parentFaultFamily, parentFaultMember, parentFaultCode, configuration);
			
			childAlarmDefinition = 
				getAlarmDefinition(childFaultFamily, childFaultMember, childFaultCode, configuration);
			
			newReductionLink.setAlarmDefinitionByParentalarmdefid(parentAlarmDefinition);
			newReductionLink.setAlarmDefinitionByChildalarmdefid(childAlarmDefinition);
			newReductionLink.getConfiguration().addReductionLinkToReductionLinks(newReductionLink);
			parentAlarmDefinition.addReductionLinkToReductionLinksForParentalarmdefid(newReductionLink);
			reductionThreshold.setAlarmDefinition(parentAlarmDefinition);
			reductionThreshold.setConfiguration(configuration);
			configuration.getReductionThresholds().add(reductionThreshold);
			parentAlarmDefinition.setReductionThreshold(reductionThreshold);
			childAlarmDefinition.addReductionLinkToReductionLinksForChildalarmdefid(newReductionLink);
			
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			try {
				AlarmConversationUtils.getInstance().saveOrUpdateReductionLink(newReductionLink, false);
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(childAlarmDefinition, false);
				AlarmConversationUtils.getInstance().saveOrUpdateReductionThreshold(reductionThreshold, false);
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(parentAlarmDefinition, false);
				SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(newReductionLink.getConfiguration(), true);
				newReductionLink = ReductionLinkHelper.findReductionLink(newReductionLink);
				this.notifyNewReductionLinkListeners(newReductionLink);
			} 
			catch( Exception e ) {
				rollbackChanges(newReductionLink, e);
			} 
		}
		catch(Exception ex) {
			ex.printStackTrace();
			throw new RuntimeException("Unexpected problem with wizard");
		} finally {
			shell.setCursor(null);
		}
	}
	
	private AlarmDefinition getAlarmDefinition(String familyName, String memberName, String faultCodeStr, Configuration config) throws Exception 
	{
		AlarmDefinition retVal = null;
		
		retVal = AlarmConversationUtils.getInstance().findMatchingAlarmDefinition(familyName, memberName, faultCodeStr, config);
		if(null == retVal) {
			retVal = new AlarmDefinition();
			retVal.setConfiguration(config);
			retVal.setFaultFamily(familyName);
			retVal.setFaultMember(memberName);
			retVal.setFaultCode(faultCodeStr);
		}
		else {
			retVal = AlarmDefinitionHelper.findAlarmDefinition(retVal);
		}
		
		return retVal;
	}

	private void rollbackChanges(ReductionLink newRedLink, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "<Parent alarm definition, child alarm definition> combination must be unique within configuration; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		e.printStackTrace();
		
		if(null != newRedLink)
		{
			childAlarmDefinition.getReductionLinksForParentalarmdefid().remove(newRedLink);
			parentAlarmDefinition.getReductionLinksForChildalarmdefid().remove(newRedLink);
			
			Configuration config = ConfigurationHelper.findConfiguration(parentAlarmDefinition.getConfiguration());
			config.getReductionLinks().remove(newRedLink);
			newRedLink.setAlarmDefinitionByChildalarmdefid(null);
			newRedLink.setAlarmDefinitionByParentalarmdefid(null);
			
			try {
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(this.parentAlarmDefinition, false);
				AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(this.childAlarmDefinition, false);
				SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
			} catch (Exception e1) {
				GuiUtils.showErrorDialog(window.getShell(), "Problems rolling back transaction", e1.getMessage());
				e1.printStackTrace();
			}
		}
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			setEnabled(true);
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void addNewReductionLinkListener(INewReductionLinkListener listener) {
		if(null != listener) {
			this.newReductionLinkListeners.add(listener);
		}
	}

	@Override
	public void notifyNewReductionLinkListeners(ReductionLink newReductionLink) 
	{
		for(INewReductionLinkListener listener: newReductionLinkListeners)
		{
			listener.update(newReductionLink);
		}
	}

	@Override
	public void removeNewReductionLinkListener(INewReductionLinkListener listener) {
		this.newReductionLinkListeners.remove(listener);
	}

	@Override
	public void setChildFaultCode(String code) {
		this.childFaultCode = code;
	}

	@Override
	public void setChildFaultFamily(String family) {
		this.childFaultFamily = family;
	}

	@Override
	public void setChildFaultMember(String member) {
		this.childFaultMember = member;
	}

	@Override
	public void setParentFaultCode(String code) {
		this.parentFaultCode = code;
	}

	@Override
	public void setParentFaultFamily(String family) {
		this.parentFaultFamily = family;
	}

	@Override
	public void setParentFaultMember(String member) {
		this.parentFaultMember = member;
	}

	@Override
	public void setReductionThreshold(ReductionThreshold threshold) {
		this.reductionThreshold = threshold;
	}
	
	public void setConfiguration(Configuration configuration)
	{
		this.configuration = configuration;
	}
}
