/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignLocationAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewLocationListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewLocationPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.LocationHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.wizards.AddLocationWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddLocationAction extends ConversationalAction implements IAssignLocationAttributes, INewLocationPublisher
{
	private String mnemonic;
	private String building;
	private String floor;
	private String position;
	private String room;
	private Collection<INewLocationListener> listeners = new ArrayList<INewLocationListener>();
	private Location newLocation;
	private Shell shell;
	
	private static final String ID = "add_location.action";
	
	public AddLocationAction(Shell shell, INewLocationListener listener)
	{
		this.shell = shell;
		setId(ID);
		setText("New Location");
		setToolTipText("Adds a new location");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/location-new.png" ));
		this.listeners.add(listener);
	}
	
	@Override
	public void doConversational() 
	{
		try {
			// Collect user input
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddLocationWizard wizard = new AddLocationWizard( this );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}

			newLocation = new Location();
			newLocation.setMnemonic(mnemonic);
			newLocation.setBuilding(building);
			newLocation.setFloor(floor);
			newLocation.setLocationPosition(position);
			newLocation.setRoom(room);
			
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AlarmConversationUtils.getInstance().saveOrUpdateLocation(newLocation, true);
			LocationHelper.findLocation(newLocation);
		}
		catch( Exception e ) {
			String message = CHANGES_NOT_SAVED;
			if(e instanceof ConstraintViolationException)
			{
				message = "Combination of: (Building, Floor, Room, Mnemonic, LocationPosition) must be unique; changes not saved";
			}
			else 
			{
				message = e.getMessage();
			}
			GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
			e.printStackTrace();
		} finally {
			shell.setCursor(null);
		}

	}

	@Override
	public void doPostConversational() 
	{
		this.notifyNewLocationListeners(newLocation);
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbPart, ISelection newSelection) 
	{
		setEnabled(true);
	}
	
	@Override
	public void setPosition(String position) {
		this.position = position;
	}

	@Override
	public void setMnemonic(String mnemonic) {
		this.mnemonic = mnemonic;
	}

	@Override
	public void setBuilding(String building) {
		this.building = building;
	}
	
	@Override
	public void setFloor(String floor)
	{
		this.floor = floor;
	}
	
	@Override
	public void setRoom(String room)
	{
		this.room = room;
	}

	@Override
	public void addNewLocationListener(INewLocationListener listener) {
		this.listeners.add(listener);
	}

	@Override
	public void notifyNewLocationListeners(Location loc) {
		for(INewLocationListener listener: listeners)
		{
			listener.update(loc);
		}
	}

	@Override
	public void removeNewLocationListener(INewLocationListener listener) {
		this.listeners.remove(listener);
	}
}
