/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignFaultMemberAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultMemberListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultMemberPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultMemberList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddFaultMemberWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddFaultMemberAction extends ConversationalAction implements IAssignFaultMemberAttributes, INewFaultMemberPublisher
{
	private FaultFamily currentlySelectedFamily;
	private String memberName;
	private Location location;
	private Collection<INewFaultMemberListener> newFaultMemberListeners = new ArrayList<INewFaultMemberListener>();
	private static final String ID = "add_faultmember.action";
	
	public AddFaultMemberAction(IWorkbenchWindow window, INewFaultMemberListener listener, FaultFamily selectedFamily)
	{
		this.window = window;
		setId(ID);
		setText("New Fault Member");
		setToolTipText("Adds a new Fault Member");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/faultmember-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.addNewFaultMemberListener(listener);
		this.currentlySelectedFamily = selectedFamily;
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();

		FaultMember newFaultMember = null;
		
		try {
			// Collect user input
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			currentlySelectedFamily = FaultFamilyHelper.findFaultFamily(currentlySelectedFamily);
			AddFaultMemberWizard wizard = new AddFaultMemberWizard( this, currentlySelectedFamily );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newFaultMember = new FaultMember();
			newFaultMember.setLocation(location);
			location.addFaultMemberToFaultMembers(newFaultMember);
			newFaultMember.setMemberName(memberName);
			newFaultMember.setFaultFamily(currentlySelectedFamily);
			this.currentlySelectedFamily.addFaultMemberToFaultMembers(newFaultMember);
			
			AlarmConversationUtils.getInstance().saveOrUpdateFaultMember(newFaultMember, false);
			AlarmConversationUtils.getInstance().saveOrUpdateLocation(location, false);
			AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(this.currentlySelectedFamily, true);
			newFaultMember = FaultMemberHelper.findFaultMember(newFaultMember);
			this.notifyNewFaultMemberListeners(newFaultMember);
		}
		catch( Exception e ) {
			rollbackChanges(newFaultMember, e);
		} finally {
			this.window.getShell().setCursor(null);
		}
	}

	private void rollbackChanges(FaultMember newFaultMember, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Fault member name must be unique within fault family; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newFaultMember)
		{
			currentlySelectedFamily.getFaultMembers().remove(newFaultMember);
			newFaultMember.setFaultFamily(null);
		}
		e.printStackTrace();
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   ((selection.getFirstElement() instanceof FaultFamily ) || 
					    (selection.getFirstElement() instanceof FaultMemberList)) )	
			{
				if(selection.getFirstElement() instanceof FaultFamily) {
					this.currentlySelectedFamily = (FaultFamily) selection.getFirstElement();
				} else {
					this.currentlySelectedFamily = ((FaultMemberList)selection.getFirstElement()).getFaultFamily();
				}
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void setLocation(Location location) {
		this.location = location;
	}

	@Override
	public void setMemberName(String name) {
		this.memberName = name;
	}

	@Override
	public void addNewFaultMemberListener(INewFaultMemberListener listener) {
		if(null != listener) {
			this.newFaultMemberListeners.add(listener);
		}
	}

	@Override
	public void notifyNewFaultMemberListeners(FaultMember newFaultMember) 
	{
		for(INewFaultMemberListener listener: newFaultMemberListeners)
		{
			listener.update(newFaultMember);
		}
	}

	@Override
	public void removeNewFaultMemberListener(INewFaultMemberListener listener) {
		this.newFaultMemberListeners.remove(listener);
	}
}
