/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Contact;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignFaultFamilyAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultFamilyListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultFamilyPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmCategoryHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddFaultFamilyWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddFaultFamilyAction extends ConversationalAction implements IAssignFaultFamilyAttributes, INewFaultFamilyPublisher
{
	private Configuration configuration;
	private AlarmCategory currentlySelectedCategory;
	private Contact contact;
	private String familyName;
	private String alarmSource;
	private String helpURL;
	private static final String ID = "add_faultfamily.action";
	private Collection<INewFaultFamilyListener> newFaultFamilyListeners = new ArrayList<INewFaultFamilyListener>();
	
	public AddFaultFamilyAction(IWorkbenchWindow window, INewFaultFamilyListener listener, Configuration config)
	{
		this.window = window;
		this.configuration = config;
		this.addNewFaultFamilyListener(listener);
		setId(ID);
		setText("New Fault Family");
		setToolTipText("Adds a new Fault Family");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/faultfamily-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();

		FaultFamily newFaultFamily = null; 
		
		try {
			// Collect user input
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			this.currentlySelectedCategory = AlarmCategoryHelper.findAlarmCategory(currentlySelectedCategory);
			AddFaultFamilyWizard wizard = new AddFaultFamilyWizard( this, configuration);
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}

			newFaultFamily = new FaultFamily();
			newFaultFamily.setConfiguration(configuration);
			newFaultFamily.setContact(contact);
			contact.addFaultFamilyToFaultFamilies(newFaultFamily);
			newFaultFamily.setFamilyName(familyName);
			newFaultFamily.setAlarmSource(alarmSource);
			newFaultFamily.setHelpURL(helpURL);
			newFaultFamily.addAlarmCategoryToAlarmCategories(currentlySelectedCategory);
			this.currentlySelectedCategory.addFaultFamilyToFaultFamilies(newFaultFamily);
			configuration.addFaultFamilyToFaultFamilies(newFaultFamily);
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(newFaultFamily, false);
			AlarmConversationUtils.getInstance().saveOrUpdateContact(contact, false);
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmCategory(this.currentlySelectedCategory, true);
			newFaultFamily = FaultFamilyHelper.findFaultFamily(newFaultFamily);
			this.notifyNewFaultFamilyListeners(newFaultFamily);
		}
		catch( Exception e ) {
			rollbackChanges(newFaultFamily, e);
		} finally {
			this.window.getShell().setCursor(null);
		}
	}
	
	private void rollbackChanges(FaultFamily newFf, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Family name must be unique within configuration; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newFf)
		{
			this.currentlySelectedCategory.getFaultFamilies().remove(newFf);
			this.configuration.getFaultFamilies().remove(newFf);
			newFf.setFaultFamilyId(null);
			newFf.setConfiguration(null);
		}
		e.printStackTrace();
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   (selection.getFirstElement() instanceof AlarmCategory ))	
			{
				this.currentlySelectedCategory = (AlarmCategory) selection.getFirstElement();
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}
	
	@Override
	public void setAlarmSource(String source) {
		this.alarmSource = source;
	}

	@Override
	public void setFamilyName(String name) {
		this.familyName = name;
	}

	@Override
	public void setHelpUrl(String url) {
		this.helpURL = url;
	}

	@Override
	public void setContact(Contact contact) {
		this.contact = contact;
	}
	
	public void setConfiguration(Configuration configuration) {
		this.configuration = configuration;
	}

	@Override
	public void addNewFaultFamilyListener(INewFaultFamilyListener listener) {
		if(null != listener) {
			this.newFaultFamilyListeners.add(listener);
		}
	}

	@Override
	public void notifyNewFaultFamilyListeners(FaultFamily newFaultFamily) 
	{
		for(INewFaultFamilyListener listener: newFaultFamilyListeners)
		{
			listener.update(newFaultFamily);
		}
	}

	@Override
	public void removeNewFaultFamilyListener(INewFaultFamilyListener listener) {
		this.newFaultFamilyListeners.remove(listener);
	}
}
