/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignFaultCodeAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultCodeListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewFaultCodePublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultCodeHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultFamilyHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultCodeList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddFaultCodeWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddFaultCodeAction extends ConversationalAction implements IAssignFaultCodeAttributes, INewFaultCodePublisher
{
	private FaultFamily currentlySelectedFamily;
	private static final String ID = "add_faultcode.action";
	private String action;
	private String cause;
	private String consequence;
	private String problemDescription;
	private boolean instant;
	private int priority;
	private int codeValue;
	private Collection<INewFaultCodeListener> newFaultCodeListeners = new ArrayList<INewFaultCodeListener>();
	
	public AddFaultCodeAction(IWorkbenchWindow window, INewFaultCodeListener listener, FaultFamily currentlySelectedFamily)
	{
		this.window = window;
		this.currentlySelectedFamily = currentlySelectedFamily;
		setId(ID);
		setText("New Fault Code");
		setToolTipText("Adds a new Fault Code");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/faultcode-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.addNewFaultCodeListener(listener);
	}
	
	@Override
	public void doConversational() {
		Shell shell = window.getShell();

		FaultCode newFaultCode = null;
		
		try {
			// Collect user input
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			this.currentlySelectedFamily = FaultFamilyHelper.findFaultFamily(currentlySelectedFamily);
			AddFaultCodeWizard wizard = new AddFaultCodeWizard( this, currentlySelectedFamily );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{ 
				return;
			}
			
			newFaultCode = new FaultCode();
			newFaultCode.setCodeValue(codeValue);
			newFaultCode.setAction(action);
			newFaultCode.setConsequence(consequence);
			newFaultCode.setIsInstant(instant);
			newFaultCode.setPriority(priority);
			newFaultCode.setCause(cause);
			newFaultCode.setProblemDescription(problemDescription);
			newFaultCode.setFaultFamily(currentlySelectedFamily);
			this.currentlySelectedFamily.addFaultCodeToFaultCodes(newFaultCode);
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AlarmConversationUtils.getInstance().saveOrUpdateFaultCode(newFaultCode, false);
			AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(this.currentlySelectedFamily, true);
			FaultCodeHelper.findFaultCode(newFaultCode);
			this.notifyNewFaultCodeListeners(newFaultCode);
		}
		catch( Exception e ) {
			rollbackChanges(newFaultCode, e);
		} finally {
			window.getShell().setCursor(null);
		}
	}
	
	private void rollbackChanges(FaultCode newFc, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Code value must be unique within fault family; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newFc)
		{
			this.currentlySelectedFamily.getFaultCodes().remove(newFc);
			newFc.setFaultFamily(null);
		}
		e.printStackTrace();
	}

	@Override
	public void doPostConversational() {
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   ((selection.getFirstElement() instanceof FaultFamily ) || 
					    (selection.getFirstElement() instanceof FaultCodeList)) )	
			{
				if(selection.getFirstElement() instanceof FaultFamily) {
					this.currentlySelectedFamily = (FaultFamily) selection.getFirstElement();
				} else {
					this.currentlySelectedFamily = ((FaultCodeList)selection.getFirstElement()).getFaultFamily();
				}
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void setAction(String action) {
		this.action = action;
	}

	@Override
	public void setCause(String cause) {
		this.cause = cause;
	}

	@Override
	public void setCodeValue(Integer value) {
		this.codeValue = value;
	}

	@Override
	public void setConsequence(String consequence) {
		this.consequence = consequence;
	}

	@Override
	public void setInstant(boolean instant) {
		this.instant = instant;
	}

	@Override
	public void setPriority(Integer priority) {
		this.priority = priority;
	}

	@Override
	public void setProblemDescription(String problemDescription) {
		this.problemDescription = problemDescription;
	}
	
	@Override
	public void addNewFaultCodeListener(INewFaultCodeListener listener) {
		if(null != listener) {
			this.newFaultCodeListeners.add(listener);
		}
	}

	@Override
	public void notifyNewFaultCodeListeners(FaultCode newFaultCode) {
		for(INewFaultCodeListener listener: newFaultCodeListeners)
		{
			listener.update(newFaultCode);
		}
	}

	@Override
	public void removeNewFaultCodeListener(INewFaultCodeListener listener) {
		this.newFaultCodeListeners.remove(listener);
	}
}
