/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.Location;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignDefaultMemberAttributes;
import alma.obops.tmcdb.alarms.ui.tree.helpers.DefaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.DefaultMemberList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.wizards.AddDefaultMemberWizard;

public class AddDefaultMemberAction extends ConversationalAction implements IAssignDefaultMemberAttributes 
{
	private FaultFamily currentlySelectedFamily;
	private static final String ID = "add_defaultmember.action";
	private Location location;
	
	public AddDefaultMemberAction(IWorkbenchWindow window)
	{
		this.window = window;
		setId(ID);
		setText("New Default Member");
		setToolTipText("Adds a new Default Member");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/defaultmember-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
	}
	
	@Override
	public void doConversational() {
		Shell shell = window.getShell();

		DefaultMember newDefaultMember = null;
		
		try {
			// Collect user input
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddDefaultMemberWizard wizard = new AddDefaultMemberWizard( this );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{ 
				return;
			}
			
			newDefaultMember = new DefaultMember();
			newDefaultMember.setLocation(location);
			location.addDefaultMemberToDefaultMembers(newDefaultMember);
			newDefaultMember.setFaultFamily(this.currentlySelectedFamily);
			this.currentlySelectedFamily.addDefaultMemberToDefaultMembers(newDefaultMember);

			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AlarmConversationUtils.getInstance().saveOrUpdateDefaultMember(newDefaultMember, false);
			AlarmConversationUtils.getInstance().saveOrUpdateLocation(location, false);
			AlarmConversationUtils.getInstance().saveOrUpdateFaultFamily(this.currentlySelectedFamily, true);
			DefaultMemberHelper.findDefaultMember(newDefaultMember);
		}
		catch( Exception e ) {
			rollbackChanges(newDefaultMember, e);
		} finally {
			window.getShell().setCursor(null);
		}
	}
	
	private void rollbackChanges(DefaultMember newDef, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Only one default member is allowed per fault family; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newDef)
		{
			this.currentlySelectedFamily.getDefaultMembers().remove(newDef);
			newDef.setFaultFamily(null);
			newDef.setLocation(null);
		}
		e.printStackTrace();
	}

	@Override
	public void doPostConversational() {
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   ((selection.getFirstElement() instanceof FaultFamily ) || 
					    (selection.getFirstElement() instanceof DefaultMemberList)) )	
			{
				if(selection.getFirstElement() instanceof FaultFamily) {
					this.currentlySelectedFamily = (FaultFamily) selection.getFirstElement();
				} else {
					this.currentlySelectedFamily = ((DefaultMemberList)selection.getFirstElement()).getFaultFamily();
				}
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void setLocation(Location location) {
		this.location = location;
	}
}
