/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.Contact;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignContactAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewContactListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewContactPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ContactHelper;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.wizards.AddContactWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;

public class AddContactAction extends ConversationalAction implements IAssignContactAttributes, INewContactPublisher
{
	private String contactName;
	private String email;
	private String gsm;
	private Shell shell;
	private Collection<INewContactListener> newContactListeners = new ArrayList<INewContactListener>();
	private Contact newContact;
	
	private static final String ID = "add_contact.action";
	
	public AddContactAction(Shell shell, INewContactListener listener)
	{
		this.shell = shell;
		setId(ID);
		setText("New Contact");
		setToolTipText("Adds a new contact");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/contact-new.png" ));
		this.newContactListeners.add(listener);
	}
	
	@Override
	public void doConversational() 
	{
		try {
			// Collect user input
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			AddContactWizard wizard = new AddContactWizard( this );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}

			newContact = new Contact();
			newContact.setContactName(contactName);
			newContact.setEmail(email);
			newContact.setGsm(gsm);
			
			shell.setCursor(shell.getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AlarmConversationUtils.getInstance().saveOrUpdateContact(newContact, true);
			ContactHelper.findContact(newContact);
		}
		catch( Exception e ) {
			String message = CHANGES_NOT_SAVED;
			if(e instanceof ConstraintViolationException)
			{
				message = "Contact name must be unique; changes not saved";
			}
			else 
			{
				message = e.getMessage();
			}
			GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
			e.printStackTrace();
		} finally {
			shell.setCursor(null);
		}

	}

	@Override
	public void doPostConversational() 
	{
		this.notifyNewContactListeners(newContact);
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbPart, ISelection newSelection) 
	{
		setEnabled(true);
	}
	
	@Override
	public void setGsm(String gsm) {
		this.gsm = gsm;
	}

	@Override
	public void setContactName(String name) {
		this.contactName = name;
	}

	@Override
	public void setEmail(String email) {
		this.email = email;
	}


	@Override
	public void addNewContactListener(INewContactListener listener) {
		this.newContactListeners.add(listener);
	}

	@Override
	public void notifyNewContactListeners(Contact contact) {
		for(INewContactListener listener: newContactListeners)
		{
			listener.update(contact);
		}
	}

	@Override
	public void removeNewContactListener(INewContactListener listener) {
		this.newContactListeners.remove(listener);
	}
}
