/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmDefinition;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultMember;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignAlarmDefinitionAttributes;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewAlarmDefinitionListener;
import alma.obops.tmcdb.alarms.ui.actions.listeners.INewAlarmDefinitionPublisher;
import alma.obops.tmcdb.alarms.ui.tree.helpers.AlarmDefinitionHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultCodeHelper;
import alma.obops.tmcdb.alarms.ui.tree.helpers.FaultMemberHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmDefinitionList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddAlarmDefinitionWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;

public class AddAlarmDefinitionAction extends ConversationalAction implements INewAlarmDefinitionPublisher, IAssignAlarmDefinitionAttributes
{
	private static final String ID = "add_alarmdefinition.action";
	private Collection<INewAlarmDefinitionListener> newAlarmDefinitionListeners = new ArrayList<INewAlarmDefinitionListener>();
	private FaultMember currentlySelectedMember;
	private FaultCode faultCode;
	
	public AddAlarmDefinitionAction(IWorkbenchWindow window, INewAlarmDefinitionListener listener)
	{
		this.window = window;
		setId(ID);
		setText("Alarm &Definition");
		setToolTipText("Adds a new Alarm Definition");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/alarm-definition-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
		this.addNewAlarmDefinitionListener(listener);
	}
	
	@Override
	public void doConversational() 
	{
		Shell shell = this.window.getShell();

		AlarmDefinition newAlarmDefinition = null;
		
		try {
			// Collect user input
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			currentlySelectedMember = FaultMemberHelper.findFaultMember(currentlySelectedMember);
			AddAlarmDefinitionWizard wizard = new AddAlarmDefinitionWizard( window, this, currentlySelectedMember );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newAlarmDefinition = new AlarmDefinition();
			faultCode = FaultCodeHelper.findFaultCode(faultCode);
			newAlarmDefinition.setFaultCode(faultCode.getCodeValue().toString());
			newAlarmDefinition.setFaultMember(currentlySelectedMember.getMemberName());
			newAlarmDefinition.setFaultFamily(faultCode.getFaultFamily().getFamilyName());
			Configuration config = ConfigurationHelper.findConfiguration(faultCode.getFaultFamily().getConfiguration());
			newAlarmDefinition.setConfiguration(config);
			config.addAlarmDefinitionToAlarmDefinitions(newAlarmDefinition);
			this.window.getShell().setCursor(this.window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmDefinition(newAlarmDefinition, false);
			SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
			this.notifyNewAlarmDefinitionListeners(newAlarmDefinition);
			AlarmDefinitionHelper.findAlarmDefinition(newAlarmDefinition);
		}
		catch( Exception e ) {
			rollbackChanges(newAlarmDefinition, e);
		} finally {
			this.window.getShell().setCursor(null);
		}
	}

	private void rollbackChanges(AlarmDefinition newDef, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Faultmember, faultcode combination must be unique within configuration; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		e.printStackTrace();
		
		Configuration config = ConfigurationHelper.findConfiguration(faultCode.getFaultFamily().getConfiguration());
		config.getAlarmDefinitions().remove(newDef);
		try {
			SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(config, true);
		} catch (Exception e1) {
			GuiUtils.showErrorDialog(window.getShell(), "Problems rolling back transaction", e1.getMessage());
			e1.printStackTrace();
		}
		
		if(null != newDef)
		{
			newDef.setFaultMember(null);
			newDef.setFaultCode(null);
		}
	}

	@Override
	public void doPostConversational() 
	{
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, this.window.getActivePage(), true);
		view.refreshTreeAndMaintainSelection();
		view.clearCaches();
	}

	@Override
	public void selectionChanged(IWorkbenchPart wbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser() &&
					   ((selection.getFirstElement() instanceof FaultMember ) || 
					    (selection.getFirstElement() instanceof AlarmDefinitionList)) )	
			{
				if(selection.getFirstElement() instanceof FaultMember) {
					this.currentlySelectedMember = (FaultMember) selection.getFirstElement();
				} else {
					this.currentlySelectedMember = null;
				}
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
		}
		else {
			setEnabled(false);
		}
	}

	@Override
	public void addNewAlarmDefinitionListener(INewAlarmDefinitionListener listener) {
		if(null != listener) {
			this.newAlarmDefinitionListeners.add(listener);
		}
	}

	@Override
	public void notifyNewAlarmDefinitionListeners(AlarmDefinition newAlarmDefinition) 
	{
		for(INewAlarmDefinitionListener listener: newAlarmDefinitionListeners)
		{
			listener.update(newAlarmDefinition);
		}
	}

	@Override
	public void removeNewAlarmDefinitionListener(INewAlarmDefinitionListener listener) {
		this.newAlarmDefinitionListeners.remove(listener);
	}

	@Override
	public void setFaultCode(FaultCode code) {
		this.faultCode = code;
	}

//	@Override
//	public void setFaultMember(FaultMember member) {
//		// TODO Auto-generated method stub
//		
//	}
}
