/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.tmcdb.alarms.ui.actions.add;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.hibernate.exception.ConstraintViolationException;

import alma.acs.tmcdb.AlarmCategory;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.DefaultMember;
import alma.acs.tmcdb.FaultCode;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.obops.tmcdb.alarms.ui.actions.listeners.IAssignAlarmCategoryAttributes;
import alma.obops.tmcdb.alarms.ui.tree.helpers.ConfigurationHelper;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.AlarmCategoryList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.DefaultMemberList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultCodeList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultFamilyList;
import alma.obops.tmcdb.alarms.ui.tree.typedlists.FaultMemberList;
import alma.obops.tmcdb.alarms.ui.utils.RcpUtils;
import alma.obops.tmcdb.alarms.ui.views.AlarmCategoriesView;
import alma.obops.tmcdb.alarms.ui.wizards.AddAlarmCategoryWizard;
import alma.obops.tmcdbgui.handlers.conversation.ConversationalAction;
import alma.obops.tmcdbgui.utils.GuiUtils;
import alma.obops.tmcdbgui.utils.conversation.AlarmConversationUtils;
import alma.obops.tmcdbgui.utils.conversation.SwConfigurationConversationUtils;
import alma.obops.tmcdbgui.views.providers.typedlists.AssemblyList;
import alma.obops.tmcdbgui.views.providers.typedlists.BaseElementList;
import alma.tmcdb.domain.Assembly;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

public class AddAlarmCategoryAction extends ConversationalAction implements IAssignAlarmCategoryAttributes
{
	private static final String ID = "add_alarmcategory.action";
	private Configuration configuration;
	private String path;
	private String name;
	private String categoryDescription;
	private boolean isDefault; 

	public AddAlarmCategoryAction(IWorkbenchWindow win, Configuration configuration) 
	{
		this.configuration = configuration;
		this.window = win;
		setId(ID);
		setText("New Alarm Category");
		setToolTipText("Adds a new Alarm Category");
		setImageDescriptor( RcpUtils.getImageDescriptor( "icons/alarmcategory-new.png" ));
		this.window.getSelectionService().addSelectionListener(this);
	}

	@Override
	public void doConversational() {
		Shell shell = window.getShell();

		AlarmCategory newAlarmCategory = null;
		
		try {
			// Collect user input
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			this.configuration = ConfigurationHelper.findConfiguration(configuration);
			AddAlarmCategoryWizard wizard = new AddAlarmCategoryWizard( this, configuration );
			WizardDialog dialog = new WizardDialog( shell, wizard );
			int ret = dialog.open();
			if( ret == WizardDialog.CANCEL ) 
			{
				return;
			}
			
			newAlarmCategory = new AlarmCategory();
			newAlarmCategory.setPath(path);
			newAlarmCategory.setIsDefault(isDefault);
			newAlarmCategory.setDescription(categoryDescription);
			newAlarmCategory.setAlarmCategoryName(name);
			newAlarmCategory.setConfiguration(configuration);
			
			this.configuration.addAlarmCategoryToAlarmCategories(newAlarmCategory);
			window.getShell().setCursor(window.getShell().getDisplay().getSystemCursor(SWT.CURSOR_WAIT));
			
			AlarmConversationUtils.getInstance().saveOrUpdateAlarmCategory(newAlarmCategory, false);
			SwConfigurationConversationUtils.getInstance().saveOrUpdateSwConfiguration(configuration, true);
		}
		catch( Exception e ) {
			rollbackChanges(newAlarmCategory, e);
		} finally {
			window.getShell().setCursor(null);
		}
	}
	
	private void rollbackChanges(AlarmCategory newCat, Exception e) 
	{
		String message = CHANGES_NOT_SAVED;
		if(e instanceof ConstraintViolationException)
		{
			message = "Categoryname must be unique within configuration; changes not saved";
		}
		else 
		{
			message = e.getMessage();
		}
		GuiUtils.showErrorDialog(window.getShell(), CHANGES_NOT_SAVED, message);
		if(null != newCat)
		{
			this.configuration.getAlarmCategories().remove(newCat);
			newCat.setConfiguration(null);
		}
		e.printStackTrace();
	}

	@Override
	public void doPostConversational() {
		// TODO: use observer pattern to decouple view and action
		AlarmCategoriesView view = (AlarmCategoriesView)RcpUtils.findView(AlarmCategoriesView.ID, window.getActivePage(), true);
		view.clearCaches();
		view.refreshTreeAndMaintainSelection();
	}

	@Override
	public void selectionChanged(IWorkbenchPart workbenchPart, ISelection newSelection) 
	{
		if( newSelection instanceof IStructuredSelection ) 
		{
			selection = (IStructuredSelection)newSelection;
			if( selection.size() == 1 && GuiUtils.isGodUser())	
			{
				setEnabled(true);
			}
			else {
				setEnabled(false);
			}
			updateConfiguration();
		}
		else {
			setEnabled(false);
		}
	}

	private void updateConfiguration() 
	{
		// TODO: is all of this really necessary? This may be overkill (though it won't 
		// necessarily hurt, other than the extra coupling between this class and other classes).
		Object selectedElement = selection.getFirstElement();
		if( selectedElement instanceof AlarmCategory ) {
			this.configuration = ((AlarmCategory)selection.getFirstElement()).getConfiguration();
		} else if( selectedElement instanceof AlarmCategoryList ) {
			this.configuration = ((AlarmCategoryList)selection.getFirstElement()).getConfiguration();
		} else if( selectedElement instanceof FaultFamily ) {
			this.configuration = ((FaultFamily) selectedElement).getConfiguration();
		} else if( selectedElement instanceof FaultFamilyList ) {
			this.configuration = ((FaultFamilyList) selectedElement).getConfiguration();
		} else if( selectedElement instanceof FaultMember ) {
			this.configuration = ((FaultMember) selectedElement).getFaultFamily().getConfiguration();
		} else if( selectedElement instanceof FaultMemberList ) {
			this.configuration = ((FaultMemberList) selectedElement).getConfiguration();
		} else if( selectedElement instanceof FaultCode ) {
			this.configuration = ((FaultCode) selectedElement).getFaultFamily().getConfiguration();
		} else if( selectedElement instanceof FaultCodeList ) {
			this.configuration = ((FaultCodeList) selectedElement).getConfiguration();
		} else if( selectedElement instanceof DefaultMember ) {
			this.configuration = ((DefaultMember) selectedElement).getFaultFamily().getConfiguration();
		} else if( selectedElement instanceof DefaultMemberList ) {
			this.configuration = ((DefaultMemberList) selectedElement).getConfiguration();
		} else if( selectedElement instanceof BaseElement ) {
			this.configuration = ((BaseElement) selectedElement).getConfiguration().getSwConfiguration();
		} else if( selectedElement instanceof BaseElementList ) {
			this.configuration = ((BaseElementList) selectedElement).getHwConfiguration().getSwConfiguration();
		} else if( selectedElement instanceof Assembly ) {
			this.configuration = ((Assembly) selectedElement).getConfiguration().getSwConfiguration();
		} else if( selectedElement instanceof AssemblyList ) {
			this.configuration = ((AssemblyList) selectedElement).getHwConfiguration().getSwConfiguration();
		} else if( selectedElement instanceof HwConfiguration) {
			this.configuration = ((HwConfiguration)selectedElement).getSwConfiguration();
		} 
		else if( selectedElement instanceof Configuration) {
			this.configuration = ((Configuration)selectedElement);
		} 
	}

	public void setConfiguration(Configuration conf) {
		this.configuration = conf;
	}

	@Override
	public void setDefault(boolean isDefault) {
		this.isDefault = isDefault;
	}

	@Override
	public void setCategoryName(String alarmCategoryName) {
		this.name = alarmCategoryName;
	}

	@Override
	public void setPath(String path) {
		this.path = path;
	}

	@Override
	public void setCategoryDescription(String description) {
		this.categoryDescription = description;
	}
}
