/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TestConfigurationService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.ComputerProcessorType;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerImplLang;
import alma.acs.tmcdb.EventChannel;
import alma.acs.tmcdb.LoggingConfig;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.HwConfiguration;

/**
 * @author rkurowsk, Dec 09, 2008
 * 
 */



public class TestSwConfigurationService extends TmcdbTestCase {

    // obtain this service using service factory
    private SwConfigurationService swConfigurationService;
    private ConfigurationService configurationService;
	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}
	
	public void setSwConfigurationService(
			SwConfigurationService swConfigurationService) {
		this.swConfigurationService = swConfigurationService;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	/* (non-Javadoc)
	 * @see org.springframework.test.AbstractTransactionalSpringContextTests#onSetUp()
	 */
	@Override
	protected void onSetUp() throws Exception {
		super.onSetUp();

		HwConfiguration conf = creationHelper.createConfiguration("Test");
		ComponentType compType = creationHelper.createComponentType("IDL:a/b/c:1.0");
		for(int i=0; i!=10; i++) {
			Component comp = creationHelper.createComponent("COMP-" + i, conf, compType, "urn");

			LoggingConfig lconf = new LoggingConfig();
			Container cont = new Container();
			cont.setContainerName("CONT" + i);
			cont.setImplLang(ContainerImplLang.JAVA);
			cont.setLoggingConfig(lconf);
			cont.setConfiguration(conf.getSwConfiguration());
			cont.setKeepAliveTime(0);
			cont.setRealTime(false);
			cont.setPath("OBOPS/ACC");

			Computer computer = new Computer();
			computer.setName("COMPUTER-" + i);
			computer.setNetworkName("COMPUTER-" + i + ".eso.org");
			computer.setRealTime(false);
			computer.setProcessorType(ComputerProcessorType.UNI);
			computer.setDiskless(false);
			computer.setConfiguration(conf.getSwConfiguration());

			EventChannel ec = new EventChannel();
			ec.setConfiguration(conf.getSwConfiguration());
			ec.setName("EVENTCHANNEL-" + i);
			ec.setPath("PATH-" + i);
			
			conf.getSwConfiguration().addComponentToComponents(comp);
			conf.getSwConfiguration().addContainerToContainers(cont);
			conf.getSwConfiguration().addNetworkDeviceToNetworkDevices(computer);
			conf.getSwConfiguration().addEventChannelToEventChannels(ec);
		}

		configurationService.create(conf);
	}
	
    @SuppressWarnings("unchecked")
	public void testHydrateComponents() {

    	commitAndStartNewTransaction();
    	List<HwConfiguration> confs = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(confs);
    	assertTrue(confs.size() > 0);

    	HwConfiguration conf = confs.get(0);
    	commitAndStartNewTransaction();

    	try {
    		conf.getSwConfiguration().getComponents().iterator().next().getComponentName();
    		fail("Should fail to load components");
    	} catch (Exception e) {
    		assertTrue("Exception expected", true);
    	}

    	swConfigurationService.hydrateComponents(conf.getSwConfiguration());

    	// Now it should work fine
    	assertEquals(10, conf.getSwConfiguration().getComponents().size());
    }

    @SuppressWarnings("unchecked")
	public void testHydrateContainers() {

    	commitAndStartNewTransaction();
    	List<HwConfiguration> confs = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(confs);
    	assertTrue(confs.size() > 0);

    	HwConfiguration conf = confs.get(0);
    	commitAndStartNewTransaction();

    	try {
    		conf.getSwConfiguration().getContainers().iterator().next().getContainerName();
    		fail("Should fail to load containers");
    	} catch (Exception e) {
    		assertTrue("Exception expected", true);
    	}

    	swConfigurationService.hydrateContainers(conf.getSwConfiguration());

    	// Now it should work fine
    	assertEquals(10, conf.getSwConfiguration().getContainers().size());

    }

    @SuppressWarnings("unchecked")
	public void testHydrateComputers() {

    	commitAndStartNewTransaction();
    	List<HwConfiguration> confs = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(confs);
    	assertTrue(confs.size() > 0);

    	HwConfiguration conf = confs.get(0);
    	commitAndStartNewTransaction();

    	try {
    		conf.getSwConfiguration().getNetworkDevices().iterator().next().getName();
    		fail("Should fail to load computers");
    	} catch (Exception e) {
    		assertTrue("Exception expected", true);
    	}

    	swConfigurationService.hydrateComputers(conf.getSwConfiguration());

    	// Now it should work fine
    	assertEquals(10, conf.getSwConfiguration().getNetworkDevices().size());
    }

    @SuppressWarnings("unchecked")
	public void testHydrateEventChannels() {

    	commitAndStartNewTransaction();
    	List<HwConfiguration> confs = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(confs);
    	assertTrue(confs.size() > 0);

    	HwConfiguration conf = confs.get(0);
    	commitAndStartNewTransaction();

    	try {
    		conf.getSwConfiguration().getEventChannels().iterator().next().getName();
    		fail("Should fail to load event channels");
    	} catch (Exception e) {
    		assertTrue("Exception expected", true);
    	}

    	swConfigurationService.hydrateEventChannels(conf.getSwConfiguration());

    	// Now it should work fine
    	assertEquals(10, conf.getSwConfiguration().getEventChannels().size());
    }
}
