/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TestLruTypeService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_TYPE_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_TYPE_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL1;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_DESC;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_FULLNAME;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_ICD;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_ICD_DATE;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_2;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NOTES;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN1;

import java.util.List;

import alma.acs.tmcdb.ComponentType;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;

/**
 * @author rkurowsk, Dec 12, 2008
 * 
 */



public class TestLruTypeService extends TmcdbTestCase {

	// obtain this service using service factory
	private LruTypeService lruTypeService;

	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	public void setLruTypeService(LruTypeService lruTypeService) {
		this.lruTypeService = lruTypeService;
	}

	// Basic creation test
	@SuppressWarnings("unchecked")
	public void testCreate00() {

		// Create and save an LruType using the service
		LruType lruType0 = newLruType(LRU_NAME_0);
		this.lruTypeService.create(lruType0);

		List<LruType> lruTypeOut = (List<LruType>)tmcdbDao.findAll(LruType.class);
		assertNotNull(lruTypeOut);
		assertEquals(1, lruTypeOut.size());
		assertEquals(lruType0.getName(), lruTypeOut.get(0).getName());

	}

	// Double creation test
	@SuppressWarnings("unchecked")
	public void testCreate01() { 

		// Create and save 2 LruTypes using the service
		LruType lruType0 = newLruType(LRU_NAME_0);
		this.lruTypeService.create(lruType0);

		LruType lruType1 = newLruType(LRU_NAME_1);
		this.lruTypeService.create(lruType1);

		List<LruType> lruTypeOut = (List<LruType>)tmcdbDao.findAll(LruType.class);
		assertNotNull(lruTypeOut);
		assertEquals(2, lruTypeOut.size());
		assertEquals(lruType0.getName(), lruTypeOut.get(0).getName());

	}

	// test delete
	@SuppressWarnings("unchecked")
	public void testDelete00(){
		
		LruType lruType0 = newLruType(LRU_NAME_0);
		this.lruTypeService.create(lruType0);

		commitAndStartNewTransaction();
		
		LruType lruTypeToDelete = this.lruTypeService.read(lruType0.getName());
		this.lruTypeService.delete(lruTypeToDelete);
		
		commitAndStartNewTransaction();
		
		List<LruType> lruTypeOut = (List<LruType>)tmcdbDao.findAll(LruType.class);
		assertNotNull(lruTypeOut);
		assertEquals(0, lruTypeOut.size());
	}
	
	@SuppressWarnings("unchecked")
	public void testFindAllLruTypes00(){
		
		LruType lruType0 = newLruType(LRU_NAME_0);
		this.lruTypeService.create(lruType0);

		LruType lruType1 = newLruType(LRU_NAME_1);
		this.lruTypeService.create(lruType1);

		commitAndStartNewTransaction();
		
		List<LruType> lruTypeOut = (List<LruType>)tmcdbDao.findAll(LruType.class);
		assertNotNull(lruTypeOut);
		assertEquals(2, lruTypeOut.size());
	}
	
	@SuppressWarnings("unchecked")
	public void testFindByAssociatedBaseElementType()
	{	
		LruType lruType0 = newLruType(LRU_NAME_0);
		this.lruTypeService.create(lruType0);
		commitAndStartNewTransaction();
		
		LruType lruType1 = newLruType(LRU_NAME_1);
		this.lruTypeService.create(lruType1);
		commitAndStartNewTransaction();
		
		LruType lruType2 = newLruType(LRU_NAME_2);
		this.lruTypeService.create(lruType2);
		commitAndStartNewTransaction();
		
		ComponentType compType = new ComponentType();
		compType.setIDL("IDL:alma.obops.testcase.nothing.interesting:1.0");
		tmcdbDao.create(compType);
		
		commitAndStartNewTransaction();

		AssemblyType atype1 = new AssemblyType("DGCK", "long/name/DGCK", BaseElementType.Antenna,
	            "description", "notes", compType, "productionCode", "simCode");
		atype1.setLruType(lruType0);
		tmcdbDao.create(atype1);
		commitAndStartNewTransaction();
		lruType0.addAssemblyType(atype1);
		
		
		AssemblyType atype2 = new AssemblyType("DGCK2", "long/name/DGCK2", BaseElementType.Antenna,
	            "description2", "notes2", compType, "productionCode", "simCode");
		atype2.setLruType(lruType1);
		tmcdbDao.create(atype2);
		commitAndStartNewTransaction();
		lruType1.addAssemblyType(atype2);
		
		
		AssemblyType atype3 = new AssemblyType("FE3", "long/name/FE3", BaseElementType.FrontEnd,
	            "description3", "notes3", compType, "productionCode", "simCode");
		atype3.setLruType(lruType2);
		tmcdbDao.create(atype3);
		commitAndStartNewTransaction();
		lruType2.addAssemblyType(atype3);
			
		this.lruTypeService.update(lruType0);
		this.lruTypeService.update(lruType1);
		this.lruTypeService.update(lruType2);
		this.tmcdbDao.update(atype3);
		this.tmcdbDao.update(atype2);
		this.tmcdbDao.update(atype1);
		commitAndStartNewTransaction();
		
		LruType[] lruTypeOut = this.lruTypeService.findByBaseElementStartupType(BaseElementStartupType.Antenna);
		assertNotNull(lruTypeOut);
		assertEquals(2, lruTypeOut.length);
		
		List<LruType> lruTypeOutAll = (List<LruType>)tmcdbDao.findAll(LruType.class);
		assertNotNull(lruTypeOutAll);
		assertEquals(3, lruTypeOutAll.size());
		
	}
	
	public void testFindAllAssemblyTypes00(){
		
		createAssemblyType0();
		createAssemblyType1();

		commitAndStartNewTransaction();
		
		AssemblyType[] assemblyTypes = this.lruTypeService.findAllAssemblyTypes();

		assertNotNull(assemblyTypes);
		assertEquals(2, assemblyTypes.length);
	}
	
	public void testFindAllAssemblyRoles00(){
		
		createAssemblyRole0();
		createAssemblyRole1();

		commitAndStartNewTransaction();
		
		AssemblyRole[] assemblyRoles = this.lruTypeService.findAllAssemblyRoles();

		assertNotNull(assemblyRoles);
		assertEquals(2, assemblyRoles.length);
	}
	
	private LruType newLruType(String lruName) {
		return new LruType(lruName, LRU_FULLNAME, LRU_ICD, LRU_ICD_DATE,
				LRU_DESC, LRU_NOTES);
	}
	
	// convenience methods for local use only
	private AssemblyType createAssemblyType(String assemblyName,
			String lruName, String urn, String idl, String configName) {

		HwConfiguration config = creationHelper.createConfiguration(configName);
		
    	/*Schema schema = */creationHelper.createSchema(urn, idl, config);
    	
		return creationHelper.createAssemblyType(assemblyName, lruName, idl);
		
	}
	
	private AssemblyType createAssemblyType0(){
		return createAssemblyType(ASSEMBLY_TYPE_NAME_0, LRU_NAME_0, URN0, IDL0, CONFIG_NAME0);
	}
	
	private AssemblyType createAssemblyType1(){
		return createAssemblyType(ASSEMBLY_TYPE_NAME_1, LRU_NAME_1, URN1, IDL1, CONFIG_NAME1);
	}
	
	// convenience methods for local use only
	private AssemblyStartup createAssemblyRole(String assemblyName,	String lruName, 
			String urn, String idl, String componentName, String configName, String assemblyRoleName, String startupScenarioName) {

		HwConfiguration config = creationHelper.createConfiguration(configName);
    	
		/*Schema schema = */creationHelper.createSchema(urn, idl, config);
		
		return creationHelper.createAssemblyStartup(assemblyName, lruName, urn, 
				componentName, idl, config, assemblyRoleName, startupScenarioName);
	}
	
	private AssemblyStartup createAssemblyRole0(){
		
		return createAssemblyRole(ANTENNA_NAME0, LRU_NAME_0, URN0, IDL0, 
				 COMPONENT_NAME0, CONFIG_NAME0, ASSEMBLY_ROLE_NAME_0, STARTUP_SCENARIO_NAME_0);
	}
	
	private AssemblyStartup createAssemblyRole1(){
		
		return createAssemblyRole(ANTENNA_NAME1, LRU_NAME_1, URN1, IDL1, 
				COMPONENT_NAME1, CONFIG_NAME1, ASSEMBLY_ROLE_NAME_1, STARTUP_SCENARIO_NAME_1);
	}
}
