/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.lang.reflect.Method;
import java.util.List;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.springframework.context.ConfigurableApplicationContext;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.utils.ConversationInterceptor;
import alma.obops.dam.utils.ConversationTokenProvider;
import alma.obops.dam.utils.SpringConstants;
import alma.obops.dam.utils.ConversationTokenProvider.ConversationToken;
import alma.obops.dam.utils.ConversationTokenProviderAdapter;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Test for the conversation interceptor (for hibernate conversations).
 * @author sharring
 */
public class TestConversationInterceptor extends TmcdbTestCase {

    private HwConfiguration config;
    private String origName;
    private SessionFactory sessionFactory;
    private ConversationInterceptor conversationInterceptor;
	private ConfigurationService configurationService;

	public void setSessionFactory(SessionFactory sessionFactory) {
		this.sessionFactory = sessionFactory;
	}

	public void setConversationInterceptor(
			ConversationInterceptor conversationInterceptor) {
		this.conversationInterceptor = conversationInterceptor;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	
	// do a bunch of things, within a single session, AND have the ability to suspend / resume the session
	// so that things can span "user-think-time" (say for example, a wizard where multiple steps must be
	// performed, with the entire thing at the end either committed as a chunk or rolled back as a chunk, with 
	// user pauses / think time intermingled).
//	public void testConversationInterceptor() throws SecurityException, NoSuchMethodException, IllegalArgumentException, IllegalAccessException, InvocationTargetException, InterruptedException
//	{
//		logger.info("testConversationInterceptor: starting");
//		
//		Method methodToInvoke = null;
//		
//		// do something in the session; do not close session & commit transaction (yet)
//		logger.info("testConversationInterceptor: invoking method one");
//		methodToInvoke = getClass().getMethod(DO_WORK_METHOD_ONE_NAME, (Class<?>[])null);
//		conversationInterceptor.invoke(methodToInvoke, this, null);
//		
//		// pause for a couple of seconds (e.g. to emulate user-think-time) during the suspended session
//		logger.info("testConversationInterceptor: sleeping");
//		Thread.sleep(1000);
//		
//		// now verify that the configuration's name has not yet changed in the db (though our in-memory 
//		// copy has changed).
//		logger.info("testConversationInterceptor: checking the configuration");
//		Session verifySession = sessionFactory.openSession();
//		Transaction verifyTransaction = verifySession.beginTransaction();
//		HwConfiguration cfg = (HwConfiguration)verifySession.load(HwConfiguration.class, new Long(0));
//		assertFalse(cfg.getName().equals(config.getName()));
//		logger.info("**********The configuration name is now: " + cfg.getName());
//		String origName = cfg.getName();
//		verifyTransaction.commit();
//		verifySession.close();
//		
//		// now do the remainder of the transaction by resuming the session & finally committing
//		logger.info("testConversationInterceptor: invoking method two");
//		methodToInvoke = getClass().getMethod(DO_WORK_METHOD_TWO_NAME, (Class<?>[])null);
//		conversationInterceptor.invoke(methodToInvoke, this, null);
//		
//		logger.info("testConversationInterceptor: checking the configuration again");
//		verifySession = sessionFactory.openSession();
//		verifyTransaction = verifySession.beginTransaction();
//		cfg = (HwConfiguration)verifySession.load(HwConfiguration.class, new Long(0));
//		logger.info("**********The configuration name is now: " + cfg.getName());
//		assertEquals(cfg.getName(), config.getName());
//		assertEquals(cfg.getName(), origName + ".1.2");
//		verifyTransaction.commit();
//		verifySession.close();
//	}

	
	
	@Override
	protected ConfigurableApplicationContext createApplicationContext(String[] locations) {
		try {
			CloningTestUtils.unzipSampleTmcdbDatabase();
			CloningTestUtils.untarSampleTmcdbDatabase();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return super.createApplicationContext(locations);
	}

	@Override
	protected String[] getConfigLocations() {
		return new String[] {SpringConstants.TEST_TMCDB_CONVERSATION_SPRING_CONFIG};
	}

	@Override
	protected void onSetUpInTransaction() throws Exception {
		// no-op
	}

	@Override
	protected void onTearDownInTransaction() throws Exception {
		// no-op
	}

	public void testConversationInterceptorWithDamServices() throws Exception
	{
		logger.info("starting");
		
		Method methodToInvoke = null;
		
		// do something in the session; do not close session & commit transaction (yet)
		logger.info("invoking method one");
		methodToInvoke = getClass().getMethod("getConfigFromServiceAndModifyConfigName", (Class<?>[])null);
		conversationInterceptor.invoke(methodToInvoke, this, null);
		
		// pause for a couple of seconds (e.g. to emulate user-think-time) during the suspended session
		logger.info("sleeping");
		Thread.sleep(3000);
		
		// now verify that the configuration's name has not - yet - changed in the db (though our in-memory 
		// copy was changed).
		logger.info("checking the configuration");
		Session verifySession = sessionFactory.openSession();
		Transaction verifyTransaction = verifySession.beginTransaction();
		HwConfiguration cfg = (HwConfiguration)verifySession.load(HwConfiguration.class, config.getId());
		assertFalse(cfg.getName().equals(config.getName()));
		assertEquals(cfg.getName(), origName);
		logger.info("**********The configuration name from db is now: " + cfg.getName());
		verifyTransaction.commit();
		verifySession.close();
		
		// now do the remainder of the transaction by resuming the session & finally committing,
		// which will commit the name changes to the configuration
		logger.info("invoking method two");
		methodToInvoke = getClass().getMethod("updateConfigFromService", (Class<?>[])null);
		conversationInterceptor.invoke(methodToInvoke, this, null);
		
		logger.info("checking the configuration again");
		verifySession = sessionFactory.openSession();
		verifyTransaction = verifySession.beginTransaction();
		HwConfiguration cfg2 = (HwConfiguration)verifySession.load(HwConfiguration.class, config.getId());
		String cfg2name = cfg2.getName();
		logger.info("**********The configuration name from db is now: " + cfg2name);
		assertEquals(origName + ".1.2", cfg2name);
		verifyTransaction.commit();
		verifySession.close();
	}
	
	public ConversationTokenProvider getConfigFromServiceAndModifyConfigName() 
	{
		logger.info("entering");
		// This is NOT the 'final' step for the session
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_PENDING);
		
		// get the config && save it into a local instance variable, for use by the next step in the conversation
		List<HwConfiguration> configs = configurationService.findAll();
		config = configs.get(0);
		origName = config.getName();
		
		String newName = config.getName() + ".1";
		logger.info("config id is: " + config.getId());
		config.setName(newName);
		
		logger.info(" exiting");
		return retVal;
	}
	
	public ConversationTokenProvider updateConfigFromService() 
	{
		logger.info("entering");
		// This IS the 'final' step for the session
		ConversationTokenProvider retVal = new ConversationTokenProviderAdapter(ConversationToken.CONVERSATION_COMPLETED);
		
		String newName = config.getName() + ".2";
		config.setName(newName);
		
		// schedule the configuration for update
		configurationService.update(config);
		
		logger.info("exiting");
		return retVal;
	}
}
