/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TestConfigurationService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;
import alma.acs.tmcdb.ComponentType;
import alma.obops.dam.ServiceException;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaToFrontEnd;
import alma.tmcdb.domain.AntennaToPad;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.FrontEnd;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.LruType;
import alma.tmcdb.domain.Pad;
import alma.tmcdb.domain.StartupScenario;

/**
 * @author rkurowsk, Dec 09, 2008
 * 
 */



public class TestConfigurationService extends TmcdbTestCase {

    // obtain this service using service factory
    private ConfigurationService configurationService;
    private StartupScenarioService startupScenarioService;
    private LruTypeService lruTypeService;
    private AntennaService antennaService;
	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	public void setStartupScenarioService(
			StartupScenarioService startupScenarioService) {
		this.startupScenarioService = startupScenarioService;
	}

	public void setLruTypeService(LruTypeService lruTypeService) {
		this.lruTypeService = lruTypeService;
	}

	public void setAntennaService(AntennaService antennaService) {
		this.antennaService = antennaService;
	}

	// Basic creation test
    public void testReadWithChildren00() throws ServiceException {        // Create and save an Antenna
    	
    	HwConfiguration config = creationHelper.createConfiguration(CONFIG_NAME0);
    	
		creationHelper.createSchema(URN0, "data", config );
        ComponentType ctype = creationHelper.createComponentType(IDL0);
        creationHelper.createComponent(COMPONENT_NAME0, config, ctype, URN0);
        
        // detach above objects
        commitAndStartNewTransaction();

        Antenna ant = new Antenna( ANTENNA_NAME0, AntennaType.ACA, new Coordinate(1d, 2d, 3d), 
        		new Coordinate(1d, 2d, 3d), 123d, System.currentTimeMillis(), 0, 0);
        config.addBaseElement(ant);
        this.antennaService.create( ant );                   
            
        HwConfiguration configuration = (HwConfiguration)this.configurationService.read(config.getId());

        // the endTransaction() calls around the read are necessary to test lazy loading issues.
        // if the transactions are not ended the configuration object is still attached & can lazy load it's children 
        commitAndStartNewTransaction();
        
        this.configurationService.hydrate(configuration);
        
        commitAndStartNewTransaction();
        
        assertNotNull( configuration );
        assertEquals( 1, configuration.getBaseElements().size());
        
        int numAntennas = 0;
        for(BaseElement be: configuration.getBaseElements()){
        	if(be.getType() == BaseElementType.Antenna){
        		numAntennas++;
        	}
        }
        assertEquals(1, numAntennas);

    }
 
    public void testAddBaseElementToStartupScenario00() throws ServiceException {        // Create and save an Antenna
    	
    	String startupScenarioName = "startupScenarioName00";
    	HwConfiguration config = creationHelper.createConfiguration(CONFIG_NAME0);
        creationHelper.createSchema(URN0, "data", config );
        StartupScenario startupScenario = creationHelper.createStartupScenario(startupScenarioName, config);
    	Antenna ant = creationHelper.createAntenna(ANTENNA_NAME0, AntennaType.ACA, config);
    	
        // detach above objects
        commitAndStartNewTransaction();
    	
        this.startupScenarioService.addBaseElementToStartupScenario(ant, startupScenario);
        
        commitAndStartNewTransaction();
        
        HwConfiguration conf = (HwConfiguration)this.configurationService.read(startupScenario.getConfiguration().getId());
        
        commitAndStartNewTransaction();
        
        conf = this.configurationService.reHydrateSimple(conf);
        
        assertTrue(config.getStartupScenarios().size() > 0);
        
        for(StartupScenario ss : conf.getStartupScenarios()){
        	if(ss.getName().equals(startupScenarioName)){
        		assertEquals( 1, ss.getBaseElementStartups().size());
        		for(AssemblyStartup as: ss.getAssemblyStartups()){
        			assertNotNull(as.getAssemblyRole().getAssemblyType().getLruType());
        		}
        		
        		for(BaseElementStartup be : ss.getBaseElementStartups()){
        			assertEquals( be.getBaseElement().getName(),ANTENNA_NAME0);
        		}
        		break;
        	}
        }
        
    }
    
    // utility method to set some things up in the DB prior to testing of cloning
    private HwConfiguration setUpConfigurationForCloning()
    {
    	LruType lru = null;
        HwConfiguration config = null;
        StartupScenario startup = null;
        ComponentType compType = null;
        Antenna antenna = null;
        Pad pad = null;
        BaseElementStartup baseElementStartup = null;
        AssemblyRole assemblyRole = null;
        FrontEnd frontEnd = null;
        
        config = creationHelper.createConfiguration("test");

        compType = new ComponentType();
        compType.setIDL("IDL:alma/Control/FOO:1.0");
        this.tmcdbDao.create(compType);

        lru = new LruType("lru", "lru", "icd", 0L, "", "");
        AssemblyType assemblyType = new AssemblyType("test",
        		"test",
        		BaseElementType.Antenna,
        		"",
        		"",
        		compType,
        		"productionCode",
        		"simulatedCode");
        assemblyRole = new AssemblyRole("aRole");
        assemblyType.addRole(assemblyRole);
        lru.addAssemblyType(assemblyType);
        lruTypeService.create(lru);

        antenna = new Antenna("DV01",
        		AntennaType.ACA,
        		new Coordinate(0.0, 0.0, 0.0),
        		new Coordinate(0.0, 0.0, 0.0),
        		4.5,
        		0L,
        		0, 
        		0);
        config.addBaseElement(antenna);
        pad = new Pad("PAD01", new Coordinate(0.0, 0.0, 0.0), 0L);
        config.addBaseElement(pad);

        new AntennaToPad(antenna, pad, 0L, 0L, true);

        frontEnd = new FrontEnd("AFrontEnd", 0L);
        config.addBaseElement(frontEnd);

        new AntennaToFrontEnd(antenna, frontEnd, 0L, 0L);

        startup = new StartupScenario("startup");
        config.addStartupScenario(startup);
        baseElementStartup = new BaseElementStartup(antenna, startup);
        baseElementStartup.setSimulated(false);
        startup.addBaseElementStartup(baseElementStartup);

        AssemblyStartup assemblyStartup = new AssemblyStartup(baseElementStartup, assemblyRole);
        assemblyStartup.setSimulated(false);

        this.configurationService.update(config);

        return config;
    }
    
    public void testCloneStartupScenario()
    {
    	HwConfiguration config = setUpConfigurationForCloning();
    	commitAndStartNewTransaction();

    	int preSize = config.getStartupScenarios().size();
    	StartupScenario originalStartup = config.getStartupScenarios().iterator().next();

        StartupScenario clonedStartup = startupScenarioService.cloneStartupScenario(originalStartup, null);

        // reset the clone's name to original's name, so that comparison will not
        // fail due to the difference in name.
        String newName = clonedStartup.getName();
        clonedStartup.setName(originalStartup.getName());
        
        // compare the original and the clone, for equality
        assertEquals(true, CloningTestUtils.safeEquals(originalStartup, clonedStartup));
        
        // reset the name, so that the transaction won't roll back due to constraint violations...
        clonedStartup.setName(newName);
        
        this.configurationService.update(clonedStartup.getConfiguration());
        commitAndStartNewTransaction();
        
        config = (HwConfiguration) configurationService.read(config.getId());
        int postSize = config.getStartupScenarios().size();
        assertEquals(preSize + 1, postSize); 
    }
    
    public void testCloneConfiguration()
    {
    	HwConfiguration config = setUpConfigurationForCloning();
    	commitAndStartNewTransaction();

    	// Should fail if we want to clone a configuration with an already existing name
    	try {
    		configurationService.cloneConfiguration(config, config.getName());
    		fail("Shouldn't be possible to get a cloned configuration with an already existing name");
    	} catch(ServiceException e) {
    		endTransactionAndIgnoreExceptions();
    		startNewTransaction();
    	}

    	int preSize = configurationService.findAll().size();
        HwConfiguration clonedConfig = configurationService.cloneConfiguration(config, null);
        
        int postSize = configurationService.findAll().size();
        assertEquals(preSize + 1, postSize);
        
        this.configurationService.hydrateConfigurationForCloning(config);
        this.configurationService.hydrateConfigurationForCloning(clonedConfig);
        
        // set the clone's name to the original's name, so we don't get a false failure
        // when comparing the two...
        String newName = clonedConfig.getName();
        clonedConfig.setName(config.getName());
        
        // compare the original and the clone, for equality
	CloningTestUtils.clearListOfProblems();
        CloningTestUtils.compareConfigurations(config, clonedConfig);
	String[] probs = CloningTestUtils.getListOfProblems();
	assertEquals(0, probs.length);
	CloningTestUtils.clearListOfProblems();
        
        // reset the name, so that the transaction won't roll back due to constraint violations...
        clonedConfig.setName(newName);
    }
    
    public void testConfigurationHydrate00() throws ServiceException {  
    	
    	HwConfiguration config = creationHelper.createConfiguration(CONFIG_NAME0);
        creationHelper.createSchema(URN0, "data", config );
    	creationHelper.createAssemblyStartup(ANTENNA_NAME0, LRU_NAME_0, URN0, 
    			COMPONENT_NAME0, IDL0, config, ASSEMBLY_ROLE_NAME_0, STARTUP_SCENARIO_NAME_0);
    	
        Antenna antenna = (Antenna)config.getBaseElements().iterator().next();
        
        commitAndStartNewTransaction();
        
        this.configurationService.hydrate(config);
                
        Antenna antFromConfig = (Antenna)config.getBaseElements().iterator().next();
        
        String newName = "newAntName";
        antenna.setName(newName);
        
        assertEquals(antFromConfig.getName(), antenna.getName());

    }
    
    public void testStartupHydrate00() throws ServiceException {  
    	
    	HwConfiguration config = creationHelper.createConfiguration(CONFIG_NAME0);
        creationHelper.createSchema(URN0, "data", config );
    	creationHelper.createAssemblyStartup(ANTENNA_NAME0, LRU_NAME_0, URN0, 
    			COMPONENT_NAME0, IDL0, config, ASSEMBLY_ROLE_NAME_0, STARTUP_SCENARIO_NAME_0);
    	
    	StartupScenario startupScenario = config.getStartupScenarios().iterator().next();
        
    	commitAndStartNewTransaction();
        
        this.configurationService.hydrateStartup(config);
                
        StartupScenario startupScenarioFromConfig = config.getStartupScenarios().iterator().next();
        
        String newName = "newSsName";
        startupScenario.setName(newName);
        
        assertEquals(startupScenarioFromConfig.getName(), startupScenario.getName());

    }
    
}
