/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * TestComputerService.java
 *
 * Copyright European Southern Observatory 2010
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.ComputerProcessorType;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.ContainerImplLang;
import alma.acs.tmcdb.LoggingConfig;
import alma.acs.tmcdb.NetworkDevice;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.HwConfiguration;

/**
 * @author rtobar, Feb 22, 2010
 * 
 */



public class TestComputerService extends TmcdbTestCase {

    // obtain these services using service factory
    private ComputerService computerService;
    private ConfigurationService configurationService;
	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	public void setComputerService(ComputerService computerService) {
		this.computerService = computerService;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	/* (non-Javadoc)
	 * @see org.springframework.test.AbstractTransactionalSpringContextTests#onSetUp()
	 */
	@Override
	protected void onSetUp() throws Exception {
		super.onSetUp();

		HwConfiguration conf = creationHelper.createConfiguration("Test");
		ComponentType compType = creationHelper.createComponentType("IDL:a/b/c:1.0");
		for(int i=0; i!=10; i++) {
			Component comp = creationHelper.createComponent("COMP-" + i, conf, compType, "urn");

			LoggingConfig lconf = new LoggingConfig();
			Container cont = new Container();
			cont.setContainerName("CONT" + i);
			cont.setImplLang(ContainerImplLang.JAVA);
			cont.setLoggingConfig(lconf);
			cont.setConfiguration(conf.getSwConfiguration());
			cont.setKeepAliveTime(0);
			cont.setRealTime(false);
			cont.setPath("OBOPS/ACC");
			comp.setContainer(cont);

			Computer computer = new Computer();
			computer.setName("COMPUTER-" + i);
			computer.setNetworkName("COMPUTER-" + i + ".eso.org");
			computer.setRealTime(false);
			computer.setProcessorType(ComputerProcessorType.UNI);
			computer.setDiskless(false);
			computer.addContainerToContainers(cont);
			computer.setConfiguration(conf.getSwConfiguration());
			cont.setComputer(computer);

			conf.getSwConfiguration().addComponentToComponents(comp);
			conf.getSwConfiguration().addContainerToContainers(cont);
			conf.getSwConfiguration().addNetworkDeviceToNetworkDevices(computer);
		}

		configurationService.create(conf);
	}
	
	@SuppressWarnings({ "unchecked" })
	public void testHydrateContainers() {

    	commitAndStartNewTransaction();
    	List<HwConfiguration> confs = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(confs);
    	assertTrue(confs.size() > 0);

    	HwConfiguration conf = confs.get(0);
    	Computer comp = null;
    	for(NetworkDevice nd: conf.getSwConfiguration().getNetworkDevices())
    		if( nd instanceof Computer ) {
    			comp = (Computer)nd;
    		}
    	assertNotNull(comp);
    	commitAndStartNewTransaction();

    	try {
    		comp.getContainers().size();
    		fail("Should fail to load containers");
    	} catch (Exception e) {
    		assertTrue("Exception expected", true);
    	}

    	computerService.hydrateContainers(comp);

    	// Now it should work fine
    	assertEquals(1, comp.getContainers().size());
    }

}
