/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.hibernate.criterion.MatchMode;

import alma.acs.tmcdb.BACIPropArchMech;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Container;
import alma.obops.dam.ServiceException;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Tests for the component service.
 * @author sharring
 */
public class TestComponentService extends TmcdbTestCase 
{
	// obtain these services using service factory
    private ComponentService componentService;
    private ComponentTypeService componentTypeService;
    private ConfigurationService configurationService;
	private TmcdbTestCreationHelper creationHelper;
	
	private ComponentType compType1;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}
    
	public void setComponentService(ComponentService componentService) {
		this.componentService = componentService;
	}

	public void setComponentTypeService(ComponentTypeService componentTypeService) {
		this.componentTypeService = componentTypeService;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	/* (non-Javadoc)
	 * @see org.springframework.test.AbstractTransactionalSpringContextTests#onSetUp()
	 */
	@Override
	protected void onSetUp() throws Exception 
	{
		super.onSetUp();

		HwConfiguration conf = creationHelper.createConfiguration("Test");
		compType1 = creationHelper.createComponentType("IDL:a/b/c:1.0-1");
		ComponentType compType2 = creationHelper.createComponentType("IDL:a/b/c:1.0-2");
		createComponentsWithBaciProperties(conf, compType1, compType2);
		
		HwConfiguration conf2 = creationHelper.createConfiguration("Test2");
		createComponentsWithBaciProperties(conf2, compType1, compType2);
		
		configurationService.update(conf);
	}

	private void createComponentsWithBaciProperties(HwConfiguration conf, ComponentType firstCompType, ComponentType secondCompType) 
	{
		for(int i=0; i<10; i++) 
		{
			ComponentType compType = (i %2 == 0) ? firstCompType : secondCompType;
			Component comp = creationHelper.createComponent("DV-" + i, conf, compType, "urn");
			comp.setPath("testPath-" + i);
			if(i < 5) {
				Set<BACIProperty> baciProps = new HashSet<BACIProperty>();
				BACIProperty baciProperty = new BACIProperty();
				baciProperty.setPropertyName("baciPropertyOne");
				baciProperty.setComponent(comp);
				baciProperty.setDescription("description");
				baciProperty.setFormat("format");
				baciProperty.setUnits("mm");
				baciProperty.setResolution("resolution");
				baciProperty.setArchive_priority(1);
				baciProperty.setArchive_min_int(1.0);
				baciProperty.setArchive_max_int(1.0);
				baciProperty.setMin_timer_trig(0.0);
				baciProperty.setDefault_timer_trig(1.0);
				baciProperty.setInitialize_devio(false);
				baciProperty.setDefault_value("2.0");
				baciProperty.setArchive_delta(1.1);
				baciProperty.setArchive_mechanism(BACIPropArchMech.NOTIFICATION_CHANNEL);
				baciProperty.setArchive_suppress(false);
				baciProps.add(baciProperty);
				comp.addBACIProperties(baciProps);
			}
			conf.getSwConfiguration().addComponentToComponents(comp);
		}
	}

	public void testGetDomainClass()
	{
		Class<?> clazz = componentService.getDomainClass();
		assertNotNull(clazz);
		assertEquals(clazz, Component.class);
	}

	@SuppressWarnings("unchecked")
	public void testFindByNamePrefixWithinConfiguration()
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// first search for a prefix that we don't have, expecting zero 'hits'
    	String[] antennaPrefixesPm = { "PM" };
    	List<Component> pmComponents = componentService.findByNamePrefixWithinConfiguration(antennaPrefixesPm, configuration.getSwConfiguration());
    	assertNotNull(pmComponents);
    	assertEquals(0, pmComponents.size());
    	
    	// now, search for a prefix for which there are 10 components; expecting 10 'hits'
    	String[] antennaPrefixesDv = { "DV" };
    	List<Component> dvComponents = componentService.findByNamePrefixWithinConfiguration(antennaPrefixesDv, configuration.getSwConfiguration());
    	assertNotNull(dvComponents);
    	assertEquals(10, dvComponents.size());  	
	}
	
	@SuppressWarnings("unchecked")
	public void testFindByParametersWithinConfiguration()
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// search for a component by parameters that should not match; expecting zero 'hits'
    	String[] params = { "componentName", "path" };
    	Object[] values = { "PM-1", "testPath-1" };
    	List<Component> components = componentService.findByParametersWithinConfiguration(params, values, configuration.getSwConfiguration());
    	assertNotNull(components);
    	assertEquals(0, components.size());
    	
    	// search for a component by parameters that should match; expecting zero 1 'hits'
    	String[] params2 = { "componentName", "path" };
    	Object[] values2 = { "DV-1", "testPath-1" };
    	List<Component> components2 = componentService.findByParametersWithinConfiguration(params2, values2, configuration.getSwConfiguration());
    	assertNotNull(components2);    	
	}
	
	@SuppressWarnings("unchecked")
	public void testFindByComponentTypeIdWithinConfiguration()
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// find a componentType from the tmcdb
    	List<ComponentType> componentTypes = componentTypeService.findAll();
    	assertNotNull(componentTypes);
    	assertEquals(2, componentTypes.size());

    	// get the compType
    	ComponentType compType = componentTypes.get(0);
    	assertNotNull(compType);

    	// now, search for a component by parameters that should match; expecting one 'hits'
    	List<Component> components =  componentService.findByComponentTypeIdWithinConfiguration(compType, configuration.getSwConfiguration());
    	assertNotNull(components);	
    	assertEquals(5, components.size());
	}
	
	@SuppressWarnings("unchecked")
	public void testFindAll()
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// find all of the components; expect 10 'hits'
    	List<Component> allComponents = componentService.findAll();
    	assertNotNull(allComponents);
    	assertEquals(20, allComponents.size());
	}
	
	@SuppressWarnings("unchecked")
	public void testFindAllBaciProperties()
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// get all the baci properties; expect 5 'hits'
    	List<BACIProperty> baciProperties = componentService.findAllBaciProperties();
    	assertNotNull(baciProperties);
    	assertEquals(10, baciProperties.size());
	}
	
	@SuppressWarnings("unchecked")
	public void testHydrateComponentType() 
	{
		// find the configurations beginning with the name 'Test', expecting 1 'hit'
    	commitAndStartNewTransaction();
    	List<HwConfiguration> configurations = (List<HwConfiguration>)configurationService.findByName("Test");
    	assertNotNull(configurations);
    	assertEquals(2, configurations.size());
    	
    	// get the configuration named 'Test'
    	HwConfiguration configuration = configurations.get(0);
    	assertNotNull(configuration);
    	
    	// find a component
    	String[] params2 = { "componentName", "path" };
    	Object[] values2 = { "DV-1", "testPath-1" };
    	List<Component> components = componentService.findByParametersWithinConfiguration(params2, values2, configuration.getSwConfiguration());
    	assertNotNull(components); 
    	
    	Component component = components.get(0);
    	assertNotNull(component);

    	// get the compType
    	ComponentType compType = component.getComponentType();
    	assertNotNull(compType);
    	
    	commitAndStartNewTransaction();
    	
    	// try to access the compType's fields; we *should* get exceptions as it is not hydrated
    	boolean exceptionCaught = false;
    	try {
    		compType.getIDL();
    		compType.getComponents();
    	} catch(Exception e) {
    		// expected
    		exceptionCaught = true;
    	}
    	assertTrue(exceptionCaught);
 
    	// hydrate the compType
    	componentService.hydrateComponentType(component);
    	
    	// now we should *not* get exceptions accessing the compType's fields
    	exceptionCaught = false;
    	try {
    		compType.getIDL();
    		compType.getComponents();
    	} catch(Exception e) {
    		// not expected
    		exceptionCaught = true;
    	}
    	assertFalse(exceptionCaught);
	}

	public void testCreation() {
		commitAndStartNewTransaction();

		try {
			componentService.update(new Component());
			fail("Should fail because everything is null");
		} catch(Exception e) {
			endTransactionAndIgnoreExceptions();
			startNewTransaction();
		}

		HwConfiguration conf = configurationService.findAll().get(0);
		configurationService.hydrateComponents(conf);

		Component c = CloningTestUtils.createComponent("DV666", "/CONTROL", null, conf.getSwConfiguration());
		try {
			componentService.update(c);
			fail("Should fail because Component Type is null");
		} catch(Exception e) {
			endTransactionAndIgnoreExceptions();
			startNewTransaction();
		}

		c = CloningTestUtils.createComponent("DV666", "/CONTROL", null, conf.getSwConfiguration());
		c.setComponentType(new ComponentType());
		c.getComponentType().setIDL("IDL");
		try {
			componentService.update(c);
			fail("Should fail because Component Type is new and is not stored/casacaded");
		} catch(Exception e) {
			endTransactionAndIgnoreExceptions();
			startNewTransaction();
		}

		c = CloningTestUtils.createComponent("DV666", "/CONTROL", null, conf.getSwConfiguration());
		c.setComponentType(new ComponentType());
		c.getComponentType().setIDL("IDL");
		try {
			componentTypeService.update(c.getComponentType());
			componentService.update(c);
		} catch(Exception e) {
			fail(e.getMessage());
		}
	}

	public void testCloneComponent() throws Exception {

		commitAndStartNewTransaction();

		// cannot pass both null name and path
		try {
			HwConfiguration conf = configurationService.findByName("Test", MatchMode.EXACT).get(0);
			configurationService.hydrateComponents(conf);
			Component c = conf.getComponents().iterator().next();
			componentService.cloneAndStoreComponentInConfiguration(c, conf.getSwConfiguration(), null, null);
			fail("It should fail with both arguments as null");
		} catch(RuntimeException e) {
			endTransactionAndIgnoreExceptions();
			commitAndStartNewTransaction();
		}

		// Try to clone same component to itself
		testCloneComponentFail("Coyping component with same name/path in same config", "Test", "DV-0", "testPath-0", "Test", "DV-0", "testPath-0");

		// Try to clone a component that is not associated to a container, copying to same and other configuration
		testCloneComponent("Test", "DV-0", "testPath-0", "Test",  "myname", "mypath");
		testCloneComponent("Test", "DV-0", "testPath-0", "Test",   null,    "mypath");
		testCloneComponent("Test", "DV-0", "testPath-0", "Test",  "myname",  null);
		testCloneComponent("Test", "DV-0", "testPath-0", "Test2", "myname", "mypath");
		testCloneComponent("Test", "DV-0", "testPath-0", "Test2",  null,    "mypath");
		testCloneComponent("Test", "DV-0", "testPath-0", "Test2", "myname",  null);

		// This we already cloned, should fail
		testCloneComponentFail("We already cloned with this name and path", "Test", "DV-0", "testPath-0", "Test", "myname", "mypath");	
		testCloneComponentFail("We already cloned with this name and path", "Test", "DV-0", "testPath-0", "Test2", "myname", "mypath");

		// Now try with a component that is associated with a container into the same and other configuration
		try {
			HwConfiguration conf = configurationService.findByName("Test", MatchMode.EXACT).get(0);
			Container cont = creationHelper.createContainer("DV01", "CONTROL", conf.getSwConfiguration());
			Component comp = creationHelper.createComponent("DV-container", conf, compType1, "urn");
			comp.setContainer(cont);
			comp.setPath("path");
			tmcdbDao.saveOrUpdate(comp);
			commitAndStartNewTransaction();
		} catch(Exception e) {
			fail("Failed to create container-associated component");
		}

		testCloneComponent("Test", "DV-container", "path", "Test",  "myname2", "mypath2");
		testCloneComponent("Test", "DV-container", "path", "Test",   null,    "mypath2");
		testCloneComponent("Test", "DV-container", "path", "Test",  "myname2",  null);
		testCloneComponent("Test", "DV-container", "path", "Test2", "myname2", "mypath2");
		testCloneComponent("Test", "DV-container", "path", "Test2",  null,    "mypath2");
		testCloneComponent("Test", "DV-container", "path", "Test2", "myname2",  null);
	}

	private void testCloneComponentFail(String message, String originConfigName, String compName, String compPath, String targetConfigName, String newName, String newPath) {
		try {
			testCloneComponent(originConfigName, compName, compPath, targetConfigName, newName, newPath);
			fail("Should fail: " + message);
		} catch(ServiceException e) {
			endTransactionAndIgnoreExceptions();
			commitAndStartNewTransaction();
		}
	}

	private void testCloneComponent(String originConfigName, String compName, String compPath, String targetConfigName, String newName, String newPath) {

		commitAndStartNewTransaction();

		// Read sizes from both origin and target configs
		HwConfiguration originConf = configurationService.findByName(originConfigName, MatchMode.EXACT).get(0);
		HwConfiguration targetConf = configurationService.findByName(targetConfigName, MatchMode.EXACT).get(0);
		configurationService.hydrateComponents(originConf);
		configurationService.hydrateComponents(targetConf);
		int initialOriginSize = originConf.getSwConfiguration().getComponents().size();
		int initialTargetSize = targetConf.getSwConfiguration().getComponents().size();

		// Get component to clone and read number of BACI props
		String[] params = { "componentName", "path" };
    	Object[] values = { compName, compPath };
		Component c = componentService.findByParametersWithinConfiguration(params, values, originConf.getSwConfiguration()).get(0);
		componentService.hydrateBACIProperties(c);
		int initialBACIProps = c.getBACIProperties().size();

		// Clone it
		Component clonedComp = componentService.cloneAndStoreComponentInConfiguration(c, targetConf.getSwConfiguration(), newName, newPath);
		componentService.create(clonedComp);
		commitAndStartNewTransaction();

		// Assert name, path and BACI props size
		targetConf = (HwConfiguration)configurationService.findByName(targetConfigName).get(0);
    	values = new Object[]{ compName, compPath };
		c = componentService.findByParametersWithinConfiguration(params, values, originConf.getSwConfiguration()).get(0);
		values = new Object[]{ newName != null ? newName : compName, newPath != null ? newPath : compPath};
		clonedComp = componentService.findByParametersWithinConfiguration(params, values, targetConf.getSwConfiguration()).get(0);

		// They names should match
		assertEquals( (newName != null ? newName : compName), clonedComp.getComponentName());
		assertEquals( (newPath != null ? newPath : compPath), clonedComp.getPath());

		// They should have the same BACI properties
		assertEquals(initialBACIProps, c.getBACIProperties().size());
		assertEquals(initialBACIProps, clonedComp.getBACIProperties().size());
		for(BACIProperty p1: c.getBACIProperties()) {
			boolean found = false;
			for(BACIProperty p2: clonedComp.getBACIProperties()) {
				if( p2.getPropertyName().equals(p1.getPropertyName()) ) {
					found = true;
					break;
				}
			}
			if( !found )
				fail("BACI Property '" + p1.getPropertyName() + "' is not present in cloned component");
		}

		// Target configuration should always have one more
		configurationService.hydrateComponents(targetConf);
		assertEquals(initialTargetSize + 1, targetConf.getSwConfiguration().getComponents().size());

		if( !originConfigName.equals(targetConfigName) ) {

			// Original should maintain same size if we're copying component to a different configuration			
			originConf = (HwConfiguration)configurationService.findByName(originConfigName).get(0);
			configurationService.hydrateComponents(originConf);
			assertEquals(initialOriginSize, originConf.getSwConfiguration().getComponents().size());

			// Also, the cloned component should have no container
			assertNull(clonedComp.getContainer());

		}
		else {
			// The cloned component should reside in the same container that that of the original component
			if( c.getContainer() == null )
				assertNull(clonedComp.getContainer());
			else
				assertEquals(c.getContainer().getContainerId(), clonedComp.getContainer().getContainerId());
		}

	}

}
