/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.List;

import org.springframework.context.ConfigurableApplicationContext;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.HwConfiguration;

public class TestCloneWithSampleTmcdbDatabase extends TmcdbTestCase {

	private ConfigurationService configurationService;

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

   public void testCloneHwConfiguration() {
      List<HwConfiguration> configs = this.configurationService.findAll();
      for (HwConfiguration config : configs) {
         String newName = config.getName() + " copy";
         HwConfiguration clonedConfig = this.configurationService
               .cloneConfiguration(config, newName);

         // set the name of the clone to the name of the orig, so that the
         // comparison does not fail for unequal names
         clonedConfig.setName(config.getName());

         // hydrate everything, so we can make the comparison w/o lazy
         // initialization exceptions from hibernate.
         configurationService.hydrateConfigurationForCloning(config);
         configurationService.hydrateConfigurationForCloning(clonedConfig);

         // now compare the original & the clone, for equality
         CloningTestUtils.clearListOfProblems();
         CloningTestUtils.compareConfigurations(config, clonedConfig);
         String[] probs = CloningTestUtils.getListOfProblems();
         assertEquals(0, probs.length);
         CloningTestUtils.clearListOfProblems();
         
         // reset the clone's name to something unique; else we'd have
         // uniqueness constraint violations when inserting into the db via hibernate/spring
         // upon commit.
         clonedConfig.setName(newName);

         // verify that the cloned config was saved into the DB
         List<HwConfiguration> allConfigs = configurationService.findAll();
         assertEquals(configs.size() + 1, allConfigs.size());
      }
   }

	@Override
	protected ConfigurableApplicationContext createApplicationContext(String[] locations) {
		try {
			CloningTestUtils.unzipSampleTmcdbDatabase();
			CloningTestUtils.untarSampleTmcdbDatabase();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return super.createApplicationContext(locations);
	}

	@Override
	protected void onSetUpInTransaction() throws Exception {
		// no-op
	}

	@Override
	protected void onTearDownInTransaction() throws Exception {
		// no-op
	}

}
