/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.HashSet;
import java.util.Set;

import alma.acs.tmcdb.BACIPropArchMech;
import alma.acs.tmcdb.BACIProperty;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.obops.dam.utils.EmptyProgressMonitor;
import alma.tmcdb.domain.HwConfiguration;

/**
 * Tests for the component service.
 * @author sharring
 */
public class TestBACIPropertyService extends TmcdbTestCase
{
	// obtain these services using service factory
    private BACIPropertyService baciService;
    private ConfigurationService configurationService;
	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	public void setBaciService(BACIPropertyService baciService) {
		this.baciService = baciService;
	}

	public void setConfigurationService(ConfigurationService configurationService) {
		this.configurationService = configurationService;
	}

	/* (non-Javadoc)
	 * @see org.springframework.test.AbstractTransactionalSpringContextTests#onSetUp()
	 */
	@Override
	protected void onSetUp() throws Exception
	{
		super.onSetUp();

		HwConfiguration conf = creationHelper.createConfiguration("Test");
		ComponentType compType1 = creationHelper.createComponentType("IDL:a/b/c:1.0-1");
		ComponentType compType2 = creationHelper.createComponentType("IDL:a/b/c:1.0-2");
		createComponentsWithBaciProperties(conf, compType1, compType2);

		HwConfiguration conf2 = creationHelper.createConfiguration("Test2");
		createComponentsWithBaciProperties(conf2, compType1, compType2);

		configurationService.update(conf);
	}

	private void createComponentsWithBaciProperties(HwConfiguration conf, ComponentType compType1, ComponentType compType2)
	{
		for(int i=0; i<10; i++)
		{
			ComponentType compType = (i %2 == 0) ? compType1 : compType2;
			Component comp = creationHelper.createComponent("DV-" + i, conf, compType, "urn");
			comp.setPath("testPath-" + i);
			if(i < 5) {
				Set<BACIProperty> baciProps = new HashSet<BACIProperty>();
				BACIProperty baciProperty = new BACIProperty();
				baciProperty.setPropertyName("baciPropertyOne");
				baciProperty.setComponent(comp);
				baciProperty.setDescription("description");
				baciProperty.setFormat("format");
				baciProperty.setUnits("mm");
				baciProperty.setResolution("resolution");
				baciProperty.setArchive_priority(1);
				baciProperty.setArchive_min_int(1.0);
				baciProperty.setArchive_max_int(1.0);
				baciProperty.setArchive_mechanism(BACIPropArchMech.NOTIFICATION_CHANNEL);
				baciProperty.setArchive_suppress(false);
				baciProperty.setMin_timer_trig(0.0);
				baciProperty.setDefault_timer_trig(1.0);
				baciProperty.setInitialize_devio(false);
				baciProperty.setDefault_value("2.0");
				baciProperty.setArchive_delta(1.1);
				baciProps.add(baciProperty);
				comp.addBACIProperties(baciProps);
			}
			conf.getSwConfiguration().addComponentToComponents(comp);
		}
	}

	public void testGetDomainClass()
	{
		Class<?> clazz = baciService.getDomainClass();
		assertNotNull(clazz);
		assertEquals(clazz, BACIProperty.class);
	}

	public void testbulkUpdateBACIProperties() {

		// Bulk change over all the properties
		BACIProperty[] allProps = baciService.findAll().toArray(new BACIProperty[0]);
		baciService.bulkUpdateBACIProperties(allProps,
				new String[]{ "format", "units" },
				new Object[] { "format2" , "units2" },
				new EmptyProgressMonitor());

		allProps = baciService.findAll().toArray(new BACIProperty[0]);
		for (int i = 0; i < allProps.length; i++) {
			assertEquals("format2", allProps[i].getFormat());
			assertEquals("units2", allProps[i].getUnits());
		}
	}
}
