/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.PAD_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.PAD_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.PAD_NAME2;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Set;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.Pad;

/**
 * 
 * @author amchavan, Sep 11, 2007
 */



public class TestPadDao extends TmcdbTestCase {
    
	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}
	
	// Basic creation test
    public void testCreate00() 
        throws SQLException {
        
        // Create and save a Pad
        Pad pad = createPad(PAD_NAME0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really there
        String sql = "select BASEELEMENTID from Pad";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        Long id = rs.getLong( 1 );
        assertEquals(pad.getId(), id );
    }  

    // Basic creation+deletion test
    public void testDelete00() 
        throws SQLException {

        // Create and save a Pad
    	Pad pad = createPad(PAD_NAME0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();

        Pad out = (Pad) tmcdbDao.read( pad.getId(), Pad.class );
        tmcdbDao.delete( out );
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from Pad";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals( 0, count );
    }

    // Double creation+deletion test
    public void testDelete02() 
        throws SQLException {
        
        // Create + save two Pads
    	Pad pad0 = createPad(PAD_NAME0, CONFIG_NAME0);
    	Pad pad1 = createPad(PAD_NAME1, CONFIG_NAME1);
    	
        commitAndStartNewTransaction();

        Pad out0 = (Pad) tmcdbDao.read( pad0.getId(), Pad.class);
        Pad out1 = (Pad) tmcdbDao.read( pad1.getId(), Pad.class );
        tmcdbDao.delete( out0 );
        tmcdbDao.delete( out1 );

        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from Pad";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals( 0, count );
    }

    // Test for findAll()
    @SuppressWarnings("unchecked")
	public void testFind00(){
        
        // Create 3 Pads
    	Pad pad0 = createPad(PAD_NAME0, CONFIG_NAME0);
    	Pad pad1 = createPad(PAD_NAME1, CONFIG_NAME1);
    	Pad pad2 = createPad(PAD_NAME2, CONFIG_NAME2);

    	commitAndStartNewTransaction();
        
        List<Pad> out = (List<Pad>)tmcdbDao.findAll(Pad.class);
        assertEquals( 3, out.size() );
        assertEquals( pad0.getId(), out.get( 0 ).getId() );
        assertEquals( pad1.getId(), out.get( 1 ).getId() );
        assertEquals( pad2.getId(), out.get( 2 ).getId() );
    }

    // Test for findAll()
    @SuppressWarnings("unchecked")
	public void testFindByName00(){
        
        // Create 3 Pads
    	Pad pad0 = createPad(PAD_NAME0, CONFIG_NAME0);
    	Pad pad1 = createPad(PAD_NAME1, CONFIG_NAME1);
    	Pad pad2 = createPad(PAD_NAME2, CONFIG_NAME2);
        commitAndStartNewTransaction();

        String subname = PAD_NAME0.substring( 0, 3 );
        List<Pad> out = (List<Pad>)tmcdbDao.findByName( subname, Pad.class );
        assertEquals( 3, out.size() );
        assertEquals( pad0.getId(), out.get( 0 ).getId() );
        assertEquals( pad1.getId(), out.get( 1 ).getId() );
        assertEquals( pad2.getId(), out.get( 2 ).getId() );
        
        out = (List<Pad>)tmcdbDao.findByName( PAD_NAME2, Pad.class);
        assertEquals( 1, out.size() );
        assertEquals( pad2.getId(), out.get( 0 ).getId() );
    }

    // Basic create+update+read test
    public void testUpdate00(){

        // Create and save an Pad
    	Pad pad = createPad(PAD_NAME0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();

        long id = pad.getId();
        Pad be = (Pad) tmcdbDao.read( id, Pad.class);
        
        // keep track of its old name
        String nameBefore = be.getName();
        
        // put in a new name and update
        final String newName = "new name";
        be.setName( newName );
        tmcdbDao.update( be );
        
        // read that back in and check
        Pad after = (Pad) tmcdbDao.read( id , Pad.class );
        String descAfter = after.getName();
        assertEquals( newName, descAfter );
        assertEquals( PAD_NAME0, nameBefore );
    }

    // Association create+read test
    public void testAssociation00(){
        
        // Create and save an Pad
    	Pad pad = createPad(PAD_NAME0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();

        long id = pad.getId();
        Pad be = (Pad) tmcdbDao.read( id , Pad.class );
        assertNotNull( be.getConfiguration() );
    }

    // Reverse association create+read test
    public void testAssociation01(){

        // Create + save two Pads
    	Pad pad0 = createPad(PAD_NAME0, CONFIG_NAME0);

    	// use same configuration so both pads belong to same one
		createPad(PAD_NAME1, pad0.getConfiguration());
        
        commitAndStartNewTransaction();
        
        long id = pad0.getConfiguration().getId();
        HwConfiguration confOut =
            (HwConfiguration) tmcdbDao.read( id , HwConfiguration.class );
        assertNotNull( confOut );
        
        Set<BaseElement> beOut = confOut.getBaseElements();
        assertNotNull( beOut );
        assertEquals( 2, beOut.size() );
//        assertEquals( pad0.getId(), beOut.get(0).getId() );
//        assertEquals( pad1.getId(), beOut.get(1).getId() );
    }
    


	// convenience method for local use only
	private Pad createPad(String padName, String configName) {
		
		HwConfiguration configuration = creationHelper.createConfiguration(configName);
		return createPad(padName, configuration);
		
	}

	// convenience method for local use only
	private Pad createPad(String padName, HwConfiguration configuration) {
		
		return creationHelper.createPad(padName,  configuration);
		
	}

}
