/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_2;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.LruType;

/**
 * 
 * @author rkurowsk, Dec 12, 2008
 */


public class TestLruTypeDao extends TmcdbTestCase {

	private static final String SELECT_LRU_COUNT = "select count(*) from lruType";
	private static final String SELECT_LRU_NAME = "select lruName from lruType";

	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	// Basic creation test
	public void testCreate00() throws SQLException {

		// Create and save an LruType
		createLruType(LRU_NAME_0);

		commitAndStartNewTransaction();

		// do an independent query on the DB to see if it's really there
		ResultSet rs = hibernateUtils.query(SELECT_LRU_NAME);
		rs.next();
		String urn = rs.getString(1);
		assertEquals(LRU_NAME_0, urn);
	}

	// Basic creation+deletion test
	public void testDelete00() throws SQLException {

		// Create and save an LruType
		LruType lruType0 = createLruType(LRU_NAME_0);

		commitAndStartNewTransaction();

		LruType out = (LruType)this.tmcdbDao.read(lruType0.getName(), LruType.class);
		this.tmcdbDao.delete(out);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see if it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_LRU_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Double creation+deletion test
	public void testDelete02() throws SQLException {

		// Create and save an LruType
		LruType lruType0 = createLruType(LRU_NAME_0);

		LruType lruType1 = createLruType(LRU_NAME_1);

		commitAndStartNewTransaction();

		LruType out0 = (LruType)this.tmcdbDao.read(lruType0.getName(), LruType.class);
		this.tmcdbDao.delete(out0);

		LruType out1 = (LruType)this.tmcdbDao.read(lruType1.getName(), LruType.class);
		this.tmcdbDao.delete(out1);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see whether it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_LRU_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Test for findAll()
	@SuppressWarnings("unchecked")
	public void testFind00() {

		// Create 3 LruTypes
		LruType lruType0 = createLruType(LRU_NAME_0);
		LruType lruType1 = createLruType(LRU_NAME_1);
		LruType lruType2 = createLruType(LRU_NAME_2);

		commitAndStartNewTransaction();

		List<LruType> out = (List<LruType>)this.tmcdbDao.findAll(LruType.class);
		assertEquals(3, out.size());
		assertEquals(lruType0.getName(), out.get(0).getName());
		assertEquals(lruType1.getName(), out.get(1).getName());
		assertEquals(lruType2.getName(), out.get(2).getName());
	}

	// Basic create+update+read test
	public void testUpdate00() {

		// Create and save an LruType
		LruType lruType0 = createLruType(LRU_NAME_0);

		commitAndStartNewTransaction();

		String descriptionForUpdate = "description after update";
		
		LruType lruTypeForUpdate = (LruType)this.tmcdbDao.read(lruType0.getName(), LruType.class);
		lruTypeForUpdate.setDescription(descriptionForUpdate);
		this.tmcdbDao.update(lruTypeForUpdate);
		
		commitAndStartNewTransaction();

		// read that back in and check
		LruType lruTypeAfterUpdate = (LruType)this.tmcdbDao.read(lruType0.getName(), LruType.class);
		String descriptionAfterUpdate = lruTypeAfterUpdate.getDescription();
		assertEquals(descriptionForUpdate, descriptionAfterUpdate);
	}

	private LruType createLruType(String lruName){
		return creationHelper.createLruType(lruName);
	}
}
