/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_DESC0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_LONGNAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Schemas;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.cloning.CloningTestUtils;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.Coordinate;
import alma.tmcdb.domain.HwConfiguration;

/**
 * 
 * @author amchavan, Sep 11, 2007
 */

public class TestConfigurationDao extends TmcdbTestCase {
    
	private TmcdbTestCreationHelper creationHelper;
    
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	// Basic creation test
    public void testCreate00() throws SQLException {
        
        // Create and save a Configuration
        HwConfiguration config = createConfiguration(CONFIG_NAME0);
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really there
        String sql = "select CONFIGURATIONID from CONFIGURATION";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        Long id = rs.getLong( 1 );
        assertEquals(config.getId(), id );
    }  

    // Basic creation+deletion test
    public void testDelete00( ) 
        throws SQLException {
        // Create, save and delete a Configuration
        HwConfiguration c = createConfiguration(CONFIG_NAME0);
        tmcdbDao.delete( c );
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from HWCONFIGURATION";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals(0, count );
    }

    // Double creation+deletion test
    public void testDelete02( ) throws  SQLException {
        
        // Create, save and delete two Configurations
        HwConfiguration c0 =  createConfiguration(CONFIG_NAME0);
        HwConfiguration c1 =  createConfiguration(CONFIG_NAME1);
        
        tmcdbDao.delete( c0 );
        tmcdbDao.delete( c1 );
        
        commitAndStartNewTransaction();

        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from HWCONFIGURATION";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals( 0, count );
    }
    
    // Test for findAll()
    @SuppressWarnings("unchecked")
	public void testFind00( ){
        
        // Create 3 Configurations
        HwConfiguration c0 =  createConfiguration(CONFIG_NAME0);
        HwConfiguration c1 =  createConfiguration(CONFIG_NAME1);
        HwConfiguration c2 =  createConfiguration(CONFIG_NAME2);

        List<HwConfiguration> out = (List<HwConfiguration>) tmcdbDao.findAll(HwConfiguration.class);
        assertEquals( 3, out.size() );
        assertEquals( c0.getId(), out.get( 0 ).getId() );
        assertEquals( c1.getId(), out.get( 1 ).getId() );
        assertEquals( c2.getId(), out.get( 2 ).getId() );
    }

    // Test for findAll()
    @SuppressWarnings("unchecked")
	public void testFindByParameter00( ){
        
        // Create 3 Configurations
        HwConfiguration c0 =  createConfiguration(CONFIG_NAME0);
        HwConfiguration c1 =  createConfiguration(CONFIG_NAME1);
        HwConfiguration c2 =  createConfiguration(CONFIG_NAME2);

        String subname = CONFIG_NAME0.substring( 0, 3 );
        List<Configuration> out = (List<Configuration>)tmcdbDao.findByParameter("configurationName", subname, Configuration.class );
        assertEquals( 3, out.size() );
        assertEquals( c0.getSwConfiguration().getConfigurationId(), out.get( 0 ).getConfigurationId() );
        assertEquals( c1.getSwConfiguration().getConfigurationId(), out.get( 1 ).getConfigurationId() );
        assertEquals( c2.getSwConfiguration().getConfigurationId(), out.get( 2 ).getConfigurationId() );
        
        out = (List<Configuration>)tmcdbDao.findByParameter( "configurationName", CONFIG_NAME2, Configuration.class );
        assertEquals( 1, out.size() );
        assertEquals( c2.getSwConfiguration().getConfigurationId(), out.get( 0 ).getConfigurationId() );
    }

    // Basic create+update+read test
    public void testUpdate00(){
        HwConfiguration c = createConfiguration(CONFIG_NAME0);
        long id = c.getId();

        // put in a new description and update
        final String newDesc = "new description";
        c.setDescription( newDesc );
        tmcdbDao.update(c.getSwConfiguration());
        tmcdbDao.update( c );
        
        commitAndStartNewTransaction();
        
        // read that back in and check
        HwConfiguration after = (HwConfiguration) tmcdbDao.read( id, HwConfiguration.class);
        String descAfter = after.getDescription();
        assertEquals( newDesc, descAfter );
        
        tmcdbDao.delete( after );
    }

	// Basic cascade creation test
    public void testCascades00() {
        
    	Configuration swCfg = new Configuration();
    	swCfg.setActive(false);
    	swCfg.setConfigurationName(CONFIG_NAME0);
    	swCfg.setFullName(CONFIG_LONGNAME0);
    	swCfg.setCreationTime(new Date());
    	swCfg.setDescription(CONFIG_DESC0);
    	
    	HwConfiguration config = new HwConfiguration(swCfg);
    	Schemas schema = new Schemas();
    	schema.setURN(URN0);
    	schema.setSchema("data");
    	schema.setConfiguration(swCfg);
    	config.getSwConfiguration().getSchemases().add(schema);
    	
    	ComponentType componentType = new ComponentType();
    	componentType.setIDL(IDL0);
    	tmcdbDao.create(componentType);

    	Component component = CloningTestUtils.createComponent(COMPONENT_NAME0, "path", componentType, config.getSwConfiguration());
		config.getSwConfiguration().getComponents().add(component);
    	
    	Antenna antenna = new Antenna(ANTENNA_NAME0, AntennaType.AEC, new Coordinate(1d, 2d, 3d), 
    			new Coordinate(1d, 2d, 3d), 123d, System.currentTimeMillis(), 0, 0);
    	config.addBaseElement(antenna);

    	config.setDescription("desc");
    	config.getBaseElements().iterator().next().setName("NewName");

    	tmcdbDao.create(componentType);
    	tmcdbDao.create(config.getSwConfiguration());
    	tmcdbDao.create(config);
    	
//    	commitAndStartNewTransaction();
//    	
//    	tmcdbDao.delete(config );
    }
    
    // convenience method for local use only 
    private HwConfiguration createConfiguration(String configurationName){
    	return this.creationHelper.createConfiguration(configurationName);
    }
}




