/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL1;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL2;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN1;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN2;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Set;

import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.ComponentType;
import alma.acs.tmcdb.Configuration;
import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.HwConfiguration;

/**
 * 
 * @author amchavan, Sep 11, 2007
 */



public class TestComponentDao extends TmcdbTestCase {

	private TmcdbTestCreationHelper creationHelper;

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	// Basic creation test
    public void testCreate00() 
        throws SQLException {
        
        // Create and save a Component
        Component comp =  createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really there
        String sql = "select COMPONENTID from COMPONENT";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        Integer id = rs.getInt( 1 );
        assertEquals( comp.getComponentId(), id );
    }  

    // Basic creation+deletion test
    public void testDelete00( ) 
        throws SQLException {
        
        // Create and save a Component
    	Component comp = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
    	
        commitAndStartNewTransaction();

        Component out = (Component) tmcdbDao.read( comp.getComponentId(), Component.class );
        tmcdbDao.delete( out );
        
        commitAndStartNewTransaction();
        
        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from COMPONENT";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals( 0, count );
    }

    // Double creation+deletion test
    public void testDelete02() 
        throws SQLException {
        
        // Create, save and delete two Components
        Component comp0 = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
        Component comp1 = createComponent(COMPONENT_NAME1, URN1, IDL1, CONFIG_NAME1);
        
        commitAndStartNewTransaction();

        Component out0 = (Component) tmcdbDao.read( comp0.getComponentId(), Component.class );
        Component out1 = (Component) tmcdbDao.read( comp1.getComponentId(), Component.class );
        tmcdbDao.delete( out0 );
        tmcdbDao.delete( out1 );
        
        commitAndStartNewTransaction();

        // Now do an independent query on the database to see whether it's
        // really not there anymore
        String sql = "select count(*) from COMPONENT";
        ResultSet rs = hibernateUtils.query( sql );
        rs.next();
        int count = rs.getInt( 1 );
        assertEquals( 0, count );
    }
    
    // Test for findAll()
    @SuppressWarnings("unchecked")
	public void testFind00(){
        
        // Create 3 Components
    	Component comp0 = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
    	Component comp1 = createComponent(COMPONENT_NAME1, URN1, IDL1, CONFIG_NAME1);     
    	Component comp2 = createComponent(COMPONENT_NAME1, URN2, IDL2, CONFIG_NAME2);     

        commitAndStartNewTransaction();
        
        List<Component> out = (List<Component>)tmcdbDao.findAll(Component.class );
        assertEquals( 3, out.size() );
        assertEquals( comp0.getComponentId(), out.get( 0 ).getComponentId() );
        assertEquals( comp1.getComponentId(), out.get( 1 ).getComponentId() );
        assertEquals( comp2.getComponentId(), out.get( 2 ).getComponentId() );
    }

    // Test for findAll()
    @SuppressWarnings("unchecked")
    public void testFindByParameter00( ){
        
        // Create 3 Components
    	Component comp0 = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
    	Component comp1 = createComponent(COMPONENT_NAME1, URN1, IDL1, CONFIG_NAME1);     
    	Component comp2 = createComponent(COMPONENT_NAME2, URN2, IDL2, CONFIG_NAME2);    
        
        commitAndStartNewTransaction();

        String subname = COMPONENT_NAME0.substring( 0, 3 );
        List<Component> out = (List<Component>)tmcdbDao.findByParameter( "componentName", subname , Component.class);
        assertEquals( 3, out.size() );
        assertEquals( comp0.getComponentId(), out.get( 0 ).getComponentId() );
        assertEquals( comp1.getComponentId(), out.get( 1 ).getComponentId() );
        assertEquals( comp2.getComponentId(), out.get( 2 ).getComponentId() );
        
        out = (List<Component>)tmcdbDao.findByParameter( "componentName", COMPONENT_NAME2 , Component.class);
        assertEquals( 1, out.size() );
        assertEquals(  comp2.getComponentId(), out.get( 0 ).getComponentId() );
    }
    
    // Test for read(id)
    public void testRead00( ){

        // Create and save a Component
    	Component comp = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
    	
        commitAndStartNewTransaction();
        
        Component out = (Component) tmcdbDao.read( comp.getComponentId(), Component.class );
        assertNotNull( out );
    }

    // Basic create+update+read test
    public void testUpdate00( ){
        
        // Create and save a Component
    	Component comp = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();

        int id = comp.getComponentId();
        Component be = (Component) tmcdbDao.read( id , Component.class );
        
        // keep track of its old name
        String nameBefore = be.getComponentName();
        
        // put in a new name and update
        final String newName = "new name";
        be.setComponentName( newName );
        tmcdbDao.update( be );
        
        // read that back in and check
        Component after = (Component) tmcdbDao.read( id , Component.class );
        String descAfter = after.getComponentName();
        assertEquals( newName, descAfter );
        assertEquals( COMPONENT_NAME0, nameBefore );
    }

    // Association create+read test
    public void testAssociation00( ){
        
        // Create and save a Component
    	Component comp = createComponent(COMPONENT_NAME0, URN0, IDL0, CONFIG_NAME0);
        
        commitAndStartNewTransaction();

        int id = comp.getComponentId();
        Component be = (Component) tmcdbDao.read( id , Component.class );
        Configuration confOut = be.getConfiguration();
        assertNotNull( confOut );
    }

    // Reverse association create+read test
    public void testAssociation01(){
        
        // Create 2 components
    	
    	HwConfiguration configuration = creationHelper.createConfiguration(CONFIG_NAME0);
    	
    	/*Schema schema = */creationHelper.createSchema(URN0, "data", configuration);
    	
    	ComponentType componentType = creationHelper.createComponentType(IDL0);
    	
        Component comp0 = creationHelper.createComponent(COMPONENT_NAME0, configuration, componentType, URN0 );
        Component comp1 = creationHelper.createComponent(COMPONENT_NAME1, configuration, componentType, URN0 );
        configuration.getSwConfiguration().getComponents().add( comp0 );
        configuration.getSwConfiguration().getComponents().add( comp1 );
        
        tmcdbDao.update( configuration );
        
        commitAndStartNewTransaction();
        
        long id = configuration.getId();
        HwConfiguration confOut = 
            (HwConfiguration) tmcdbDao.read( id , HwConfiguration.class);
        assertNotNull( confOut );
        
        Set<Component> beOut = confOut.getComponents();
        assertNotNull( beOut );
        assertEquals( 2, beOut.size() );
    }
    
    // convenience method for local use only 
    private Component createComponent(String componentName, String urn, String idl, String configName){
    	
    	HwConfiguration config = creationHelper.createConfiguration(configName);
    	
    	/*Schema schema = */creationHelper.createSchema(urn, idl, config);
    	
    	ComponentType componentType = creationHelper.createComponentType(idl);
    	
    	return creationHelper.createComponent(componentName, config, componentType, urn);
    	
    }
}
