/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/*
 * TestAssemblyStartupDao
 * 
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.ASSEMBLY_ROLE_NAME_2;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.CONFIG_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL1;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL2;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.LRU_NAME_2;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_0;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_1;
import static alma.obops.dam.testutils.TmcdbTestConstants.STARTUP_SCENARIO_NAME_2;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN1;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN2;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.AssemblyStartup;
import alma.tmcdb.domain.HwConfiguration;

/**
 * 
 * @author rkurowsk, Dec 15, 2008
 */



public class TestAssemblyStartupDao extends TmcdbTestCase {

	private TmcdbTestCreationHelper creationHelper;
	
	private static final String SELECT_ASSEMBLY_STARTUP_COUNT = "select count(*) from assemblyStartup";
	private static final String SELECT_ASSEMBLY_STARTUP_ID = "select assemblyStartupId from assemblyStartup";

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}
	
	// Basic creation test
	public void testCreate00() throws SQLException {

		// Create and save an AssemblyStartup
		AssemblyStartup as = createAssemblyStartup0();

		commitAndStartNewTransaction();

		// do an independent query on the DB to see if it's really there
		ResultSet rs = hibernateUtils.query(SELECT_ASSEMBLY_STARTUP_ID);
		rs.next();
		
		//  assemblyId, startupId, baseElementId
		Long assemblyStartupId = rs.getLong(1);
		
		assertEquals(as.getId(), assemblyStartupId);
	}

	// Basic creation+deletion test
	public void testDelete00() throws SQLException {

		// Create and save an AssemblyStartup
		AssemblyStartup assemblyStartup = createAssemblyStartup0();

		commitAndStartNewTransaction();

		AssemblyStartup out = (AssemblyStartup) this.tmcdbDao.read(
				assemblyStartup.getId(), AssemblyStartup.class);
		
		this.tmcdbDao.delete(out);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see if it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_ASSEMBLY_STARTUP_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Double creation+deletion test
	public void testDelete02() throws SQLException {

		AssemblyStartup assemblyStartup0 = createAssemblyStartup0();
		commitAndStartNewTransaction();
		
		AssemblyStartup assemblyStartup1 = createAssemblyStartup1();
		commitAndStartNewTransaction();
		
		AssemblyStartup out0 = (AssemblyStartup) this.tmcdbDao.read(
				assemblyStartup0.getId(), AssemblyStartup.class);
		
		this.tmcdbDao.delete(out0);

		AssemblyStartup out1 = (AssemblyStartup) this.tmcdbDao.read(
				assemblyStartup1.getId(), AssemblyStartup.class);
				
		this.tmcdbDao.delete(out1);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see whether it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_ASSEMBLY_STARTUP_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Test for findAll()
	@SuppressWarnings("unchecked")
	public void testFind00() {

		// Create 3 AssemblyStartups
		AssemblyStartup assemblyStartup0 = createAssemblyStartup0();
		AssemblyStartup assemblyStartup1 = createAssemblyStartup1();
		AssemblyStartup assemblyStartup2 = createAssemblyStartup2();

		commitAndStartNewTransaction();

		List<AssemblyStartup> out = (List<AssemblyStartup>)this.tmcdbDao.findAll(AssemblyStartup.class);
		assertEquals(3, out.size());
		assertEquals(assemblyStartup0.getId(), out.get(0).getId());
		assertEquals(assemblyStartup1.getId(), out.get(1).getId());
		assertEquals(assemblyStartup2.getId(), out.get(2).getId());
	}




	// convenience methods for local use only
	private AssemblyStartup createAssemblyStartup(String assemblyName,	String lruName, 
			String urn, String idl, String componentName, String configName, String assemblyRoleName, String startupScenarioName) {

		HwConfiguration config = creationHelper.createConfiguration(configName);
    	
		/*Schema schema = */creationHelper.createSchema(urn, idl, config);
		
		return creationHelper.createAssemblyStartup(assemblyName, lruName, urn, 
				componentName, idl, config, assemblyRoleName, startupScenarioName);
	}
	
	private AssemblyStartup createAssemblyStartup0(){
		
		return createAssemblyStartup(ANTENNA_NAME0, LRU_NAME_0, URN0, IDL0, 
				 COMPONENT_NAME0, CONFIG_NAME0, ASSEMBLY_ROLE_NAME_0, STARTUP_SCENARIO_NAME_0);
	}
	
	private AssemblyStartup createAssemblyStartup1(){
		
		return createAssemblyStartup(ANTENNA_NAME1, LRU_NAME_1, URN1, IDL1, 
				COMPONENT_NAME1, CONFIG_NAME1, ASSEMBLY_ROLE_NAME_1, STARTUP_SCENARIO_NAME_1);
	}
	
	private AssemblyStartup createAssemblyStartup2(){
		
		return createAssemblyStartup(ANTENNA_NAME2, LRU_NAME_2, URN2, IDL2, 
				COMPONENT_NAME2, CONFIG_NAME2, ASSEMBLY_ROLE_NAME_2, STARTUP_SCENARIO_NAME_2);
	}
}
