/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.tmcdb.dao;

import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.ANTENNA_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME0;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME1;
import static alma.obops.dam.testutils.TmcdbTestConstants.COMPONENT_NAME2;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL0;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL1;
import static alma.obops.dam.testutils.TmcdbTestConstants.IDL2;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN0;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN1;
import static alma.obops.dam.testutils.TmcdbTestConstants.URN2;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Set;

import alma.obops.dam.testutils.TmcdbTestCase;
import alma.obops.dam.testutils.TmcdbTestCreationHelper;
import alma.tmcdb.domain.Antenna;
import alma.tmcdb.domain.AntennaType;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

/**
 * 
 * @author amchavan, Sep 11, 2007
 */


public class TestAntennaDao extends TmcdbTestCase {

	private TmcdbTestCreationHelper creationHelper;

	private static final String SELECT_ANTENNA_COUNT = "select count(*) from Antenna";
	private static final String SELECT_ANTENNA_ID = "select BASEELEMENTID from ANTENNA";

	/*
	 * Setters for dependency injection
	 */
	public void setCreationHelper(TmcdbTestCreationHelper creationHelper) {
		this.creationHelper = creationHelper;
	}

	// Association create+read test
	public void testAssociation00() {
		// Create and save an Antenna
		Antenna ant = createAntenna0();

		commitAndStartNewTransaction();

		Antenna be = (Antenna) tmcdbDao.read(ant.getId(), Antenna.class);
		assertNotNull(be.getConfiguration());
	}

	// Reverse association create+read test
	public void testAssociation01() {

		// Create + save two Antennas
		Antenna ant0 = createAntenna0();
		createAntenna1();

		commitAndStartNewTransaction();

		long id = ant0.getConfiguration().getId();
		HwConfiguration confOut = (HwConfiguration) tmcdbDao.read(id, HwConfiguration.class);
		assertNotNull(confOut);

		Set<BaseElement> beOut = confOut.getBaseElements();
		assertNotNull(beOut);
		assertEquals(1, beOut.size());
	}

	// Basic creation test
	public void testCreate00() throws SQLException {

		// Create and save an Antenna
		Antenna ant = createAntenna0();

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see whether it's
		// really there
		ResultSet rs = hibernateUtils.query(SELECT_ANTENNA_ID);
		rs.next();
		Long id = rs.getLong(1);
		assertEquals(ant.getId(), id);
	}

	// Basic creation+deletion test
	public void testDelete00() throws SQLException {

		// Create and save a Antenna
		Antenna ant = createAntenna0();

		commitAndStartNewTransaction();

		Antenna out = (Antenna) tmcdbDao.read(ant.getId(), Antenna.class);
		tmcdbDao.delete(out);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see whether it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_ANTENNA_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Double creation+deletion test
	public void testDelete02() throws SQLException {

		// Create + save two Antennas
		Antenna ant0 = createAntenna0();
		Antenna ant1 = createAntenna1();

		commitAndStartNewTransaction();

		Antenna out0 = (Antenna) tmcdbDao.read(ant0.getId(), Antenna.class);
		Antenna out1 = (Antenna) tmcdbDao.read(ant1.getId(), Antenna.class);
		tmcdbDao.delete(out0);
		tmcdbDao.delete(out1);

		commitAndStartNewTransaction();

		// Now do an independent query on the database to see whether it's
		// really not there anymore
		ResultSet rs = hibernateUtils.query(SELECT_ANTENNA_COUNT);
		rs.next();
		int count = rs.getInt(1);
		assertEquals(0, count);
	}

	// Test for findAll()
	@SuppressWarnings("unchecked")
	public void testFind00() {

		// Create 3 Antennas
		/*Antenna ant0 = */createAntenna0();
		/*Antenna ant1 = */createAntenna1();
		/*Antenna ant2 = */createAntenna2();

		commitAndStartNewTransaction();

		List<Antenna> out = (List<Antenna> ) tmcdbDao.findAll(Antenna.class);
		assertEquals(3, out.size());
//		assertEquals(ant0.getId(), out.get(0).getId());
//		assertEquals(ant1.getId(), out.get(1).getId());
//		assertEquals(ant2.getId(), out.get(2).getId());
	}

	// Test for findAll()
	@SuppressWarnings("unchecked")
	public void testFindByName00() {

		// Create 3 Antennas
		Antenna ant0 = createAntenna0();
		Antenna ant1 = createAntenna1();
		Antenna ant2 = createAntenna2();

		commitAndStartNewTransaction();

		String subname = ANTENNA_NAME0.substring(0, 3);
		List<Antenna> out = (List<Antenna>)tmcdbDao.findByName(subname, Antenna.class);
		assertEquals(3, out.size());
		assertEquals(ant0.getId(), out.get(0).getId());
		assertEquals(ant1.getId(), out.get(1).getId());
		assertEquals(ant2.getId(), out.get(2).getId());

		out = (List<Antenna>)tmcdbDao.findByName(ANTENNA_NAME2, Antenna.class);
		assertEquals(1, out.size());
		assertEquals(ant2.getId(), out.get(0).getId());
	}

	// Test for read(id)
	public void testRead00() {

		Antenna ant0 = createAntenna0();

		commitAndStartNewTransaction();

		Antenna out = (Antenna) tmcdbDao.read(ant0.getId(), Antenna.class);
		assertNotNull(out);
	}


	// Basic create+update+read test
	public void testUpdate00() {

		// Create and save an Antenna
		Antenna ant0 = createAntenna0();

		commitAndStartNewTransaction();

		// keep track of its old name
		String nameBefore = ant0.getName();
		Antenna before = (Antenna) tmcdbDao.read(ant0.getId(), Antenna.class);

		// put in a new name and update
		final String newName = "new name";
		before.setName(newName);
		tmcdbDao.update(before);

		// read that back in and check
		Antenna after = (Antenna) tmcdbDao.read(ant0.getId(), Antenna.class);
		String descAfter = after.getName();
		assertEquals(newName, descAfter);
		assertEquals(ANTENNA_NAME0, nameBefore);
	}

	// convenience method for local use only
	private Antenna createAntenna(String antennaName, AntennaType antennaType,
			String urn, String idl, String configName) {

		HwConfiguration config = creationHelper.createConfiguration(configName);

		/*Schema schema = */creationHelper.createSchema(urn, idl, config);

		commitAndStartNewTransaction();
		
		return creationHelper.createAntenna(antennaName, antennaType, config);
	}

	private Antenna createAntenna0() {
		return createAntenna(ANTENNA_NAME0, AntennaType.ACA, URN0, IDL0,
				COMPONENT_NAME0);
	}

	private Antenna createAntenna1() {
		return createAntenna(ANTENNA_NAME1, AntennaType.VA, URN1, IDL1,
				COMPONENT_NAME1);
	}

	private Antenna createAntenna2() {
		return createAntenna(ANTENNA_NAME2, AntennaType.AEC, URN2, IDL2,
				COMPONENT_NAME2);
	}

}
