/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/

package alma.obops.dam.utils.xstream;

import java.util.List;

import org.hibernate.criterion.MatchMode;

import alma.obops.dam.config.TmcdbContextFactory;
import alma.obops.dam.tmcdb.service.BaseElementService;
import alma.obops.dam.tmcdb.service.ConfigurationService;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.HwConfiguration;

import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.reflection.ReflectionConverter;
import com.thoughtworks.xstream.converters.reflection.ReflectionProvider;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

public class BaseElementConverter extends ReflectionConverter
{
	private static final String GLOBAL_PREFIX = "__global::";
	private HwConfiguration configBeingExported;

	public BaseElementConverter(Mapper mapper, ReflectionProvider reflectionProvider, HwConfiguration conf) 
	{
		super(mapper, reflectionProvider);
		this.configBeingExported = conf;
	}

	@Override
	public void marshal(Object obj, HierarchicalStreamWriter streamWriter,
			MarshallingContext marshallingContext) 
	{
		BaseElement be = (BaseElement) obj;
		if(!be.getConfiguration().getId().equals(configBeingExported.getId()))
		{
			// special handling of cross-config references...
			streamWriter.setValue(GLOBAL_PREFIX + configBeingExported.getGlobalConfiguration().getName() + GLOBAL_PREFIX + be.getName());
		}
		else {
			super.marshal(obj, streamWriter, marshallingContext);
		}
	}

	@Override
	public Object unmarshal(HierarchicalStreamReader streamReader,
			UnmarshallingContext unmarshallingContext) 
	{
		Object retVal = null;
		
		String value = streamReader.getValue();
		if(value.startsWith(GLOBAL_PREFIX)) 
		{
			int startIndex = GLOBAL_PREFIX.length();
			int endIndex = 	value.indexOf(GLOBAL_PREFIX, GLOBAL_PREFIX.length()); 
			String configname = value.substring(startIndex, endIndex);
			
			ConfigurationService service =  TmcdbContextFactory.INSTANCE.getConfigurationService();
			List<HwConfiguration> existingConfs = service.findByName(configname, MatchMode.EXACT);
			if(existingConfs == null || existingConfs.size() != 1) {
					throw new RuntimeException(new GlobalConfigurationUnmarshallingException("Could not locate the global configuration named '" + configname + "'"));
			}
			HwConfiguration existingConf = existingConfs.get(0);

			String bename = value.substring(endIndex + GLOBAL_PREFIX.length());
			BaseElementService beservice =  TmcdbContextFactory.INSTANCE.getBaseElementService();
			List<BaseElement> existingBes = beservice.findBaseElementByNameInConfiguration(bename, existingConf);
			if(existingBes == null || existingBes.size() != 1) {
				throw new RuntimeException(new GlobalConfigurationUnmarshallingException("Could not locate the global baseelement named '" + bename 
						+ "' in configuration named '" + configname + "'"));
			}
			BaseElement existingBe = existingBes.get(0);
			retVal = existingBe;
		}
		else 
		{
			retVal = super.unmarshal(streamReader, unmarshallingContext);
		}

		return retVal;
	}

	@SuppressWarnings("unchecked")
	@Override
	public boolean canConvert(Class clazz) {
		return BaseElement.class.isAssignableFrom(clazz);
	}
}
