/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.io.Serializable;
import java.net.URI;
import java.util.List;
import java.util.logging.Logger;

import alma.obops.dam.DaoException;
import alma.obops.dam.DataAccessObject;
import alma.obops.dam.Service;
import alma.obops.dam.ServiceException;
import alma.obops.dam.tmcdb.dao.TmcdbDao;


/**
 * The abstract root of all TmcdbService classes.
 * 
 * @author rkurowsk, Dec 18, 2009
 * 
 */



public abstract class TmcdbAbstractService implements Service {

    private TmcdbDao tmcdbDao;
    
    protected Logger logger;

    /**
     * Sets a logger in our service class
     */
    public void setLogger(Logger logger) {
    	this.logger = logger;
    }

    /**
     * Persist the input domain object; that is, create a representation 
     * of it in the underlying database
     * 
     * @return The generated database ID for that object
     * 
     * @see DataAccessObject#create(Object)
     */
    public Serializable create( Object domainObject ) throws ServiceException {
        try {
            DataAccessObject dao = this.getDao();
            return dao.create( domainObject );
        }
        catch( DaoException e ) {
            throw new ServiceException( e.getMessage(), e );
        }
    }

    /**
     * Make the input domain object transient; that is, remove its
     * representation from the underlying database
     * 
     * @see DataAccessObject#delete(Object)
     */
    public void delete( Object domainObject ) throws ServiceException {
        this.getDao().delete( domainObject );
    }

    /**
     * Retrieve a domain object from the database.
     * @param id
     *            Numerical identification for the object to retrieve
     *            
     * @return An instance of the input class, whose ID is the the input number
     *
     * @see DataAccessObject#read(Class, long)
     */
    public Object read( long id ) throws ServiceException {
        return this.getDao().read( id, getDomainClass() );
    }

    /**
     * Retrieve a domain object from the database.
     * @param id
     *            Numerical identification for the object to retrieve
     *            
     * @return An instance of the input class, whose ID is the the input number
     *
     * @see DataAccessObject#read(Class, long)
     */
    public Object read( int id ) throws ServiceException {
        return this.getDao().read( id, getDomainClass() );
    }

    /**
     * Retrieve a domain object from the database.
     * 
     * @param uid
     *            Numerical identification for the object to retrieve
     *            
     * @return An instance of the input class, whose ID is the the input number
     *
     * @see DataAccessObject#read(Class, long)
     */
    public Object read( URI uid ) throws ServiceException {
    	return this.getDao().read( uid, getDomainClass() );
    }

    /**
     * Synchronize the input domain object with the database; that is, update
     * its representation in the underlying database
     * 
     * @see DataAccessObject#update(Object)
     */
    public void update( Object domainObject ) throws ServiceException {
        this.getDao().update( domainObject );
    }

    /**
     * Re-attaches a domain object to the DB and populates it's children
     * 
     * @param domainObject
     */
    public void hydrate( Object domainObject ) throws ServiceException{
    	// only implement this method in service subclasses that depend on Hibernate
    }
    
	/* (non-Javadoc)
	 * @see alma.obops.dam.Service#findByName(java.lang.String)
	 */
	@SuppressWarnings("cast")
	public List<?> findByName(String substring) throws ServiceException {
		return (List<?>)getDao().findByName(substring, getDomainClass());
	}
    
    /**
     * Public constructor
     *
     */
    public TmcdbAbstractService(){}

    /**
     * @return The Data Access Object we use for our business
     */
    public TmcdbDao getDao(){
    	return tmcdbDao;
    }

    public void setTmcdbDao(TmcdbDao tmcdbDao){
    	this.tmcdbDao = tmcdbDao;
    }

    public List<?> find(List<Object> searchCriteria, List<Object> orderCriteria) {
        return tmcdbDao.find(searchCriteria, orderCriteria, getDomainClass());
    }

    public void runWithAttachedObject(Object o, Runnable r) {
    	getDao().reAttach(o);
    	r.run();
    }
}