/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * 
 */
package alma.obops.dam.tmcdb.service;

import alma.acs.tmcdb.AcsService;
import alma.acs.tmcdb.Component;
import alma.acs.tmcdb.Computer;
import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.Container;
import alma.acs.tmcdb.EventChannel;
import alma.acs.tmcdb.NetworkDevice;
import alma.acs.tmcdb.NotificationServiceMapping;
import alma.obops.dam.ServiceException;

/**
 * Business layer implementation for Software Configuration objects
 * @author rtobar
 *
 */
public class SwConfigurationServiceImpl extends TmcdbAbstractService implements SwConfigurationService {

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#getDomainClass()
	 */
	@Override
	public Class<?> getDomainClass() {
		return Configuration.class;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#hydrateComponents(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public void hydrateComponents(Configuration config) {
		this.getDao().reAttach(config);
		for(Component comp: config.getComponents()) {
			comp.getComponentName();
			comp.getConfiguration().getConfigurationName();
		}
	}
	
	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#hydrateComponents(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public void hydrateBACIProperties(Configuration config) {
		this.getDao().reAttach(config);
		for(Component comp: config.getComponents())
			comp.getBACIProperties().size();
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#hydrateComputers(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public void hydrateComputers(Configuration config) {
		this.getDao().reAttach(config);
		for(NetworkDevice comp: config.getNetworkDevices())
			if( comp instanceof Computer ) {
				((Computer)comp).getNetworkName();
				comp.getConfiguration().getConfigurationName();
			}
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#hydrateContainers(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public void hydrateContainers(Configuration config) {
		this.getDao().reAttach(config);
		for(Container comp: config.getContainers()) {
			comp.getContainerName();
			comp.getConfiguration().getConfigurationName();
		}
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.SwConfigurationService#hydrateEventChannels(alma.acs.tmcdb.Configuration)
	 */
	@Override
	public void hydrateEventChannels(Configuration config) {
		this.getDao().reAttach(config);
		for(EventChannel ec: config.getEventChannels())
			ec.getConnectionReliability();
	}
	
	@Override
	public void hydrateNetworkDevices(Configuration config) {
		this.getDao().reAttach(config);
		for(NetworkDevice nd: config.getNetworkDevices()) {
			nd.getNetworkName();
		}
	}

	@Override
	public void hydrateSchemas(Configuration config) {
		this.getDao().reAttach(config);
		config.getSchemases().size();
	}
	
	@Override
	public void hydrateAlarmCategories(Configuration config)
			throws ServiceException 
	{
		this.getDao().reAttach(config);
		config.getAlarmCategories().size();
	}

	@Override
	public void hydrateFaultFamilies(Configuration config)
			throws ServiceException 
	{
		this.getDao().reAttach(config);
		config.getFaultFamilies().size();
	}

	@Override
	public void hydrateReductionLinks(Configuration config)
			throws ServiceException 
	{
		this.getDao().reAttach(config);
		config.getReductionLinks().size();
	}

	@Override
	public void hydrateAlarmDefinitions(Configuration config)
			throws ServiceException 
	{
		this.getDao().reAttach(config);
		config.getAlarmDefinitions().size();	
	}

	@Override
	public void hydrateReductionThresholds(Configuration config)
			throws ServiceException 
	{
		this.getDao().reAttach(config);
		config.getReductionThresholds().size();	
	}

	@Override
	public void hydrateAcsServices(Configuration config) {
		this.getDao().reAttach(config);
		for(AcsService service: config.getAcsServices()) {
			service.getServiceInstanceName();
			service.getConfiguration().getConfigurationName();
		}
	}

	@Override
	public void hydrateNotificationServiceMappings(Configuration config) {
		this.getDao().reAttach(config);
		for(NotificationServiceMapping mapping: config.getNotificationServiceMappings()) {
			mapping.getDefaultNotificationService();
			mapping.getConfiguration().getConfigurationName();
		}
	}
	
	@Override
	public void hydrateManagers(Configuration config) {
		this.getDao().reAttach(config);
		config.getManagers().size();
	}
}
