/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * ConfigurationService.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import alma.acs.tmcdb.Configuration;
import alma.obops.dam.ServiceException;

/**
 * Business layer for Configuration
 *
 * @author amchavan, Sep 10, 2008
 * 
 */



public interface SwConfigurationService extends SearchableService {

    /**
     * @see alma.obops.dam.Service#getDomainClass()
     */
    @Override
    public Class<?> getDomainClass();

    /**
     * Hydrates all the <code>Component</code> objects associated with a given configuration.
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated components
     * 
     * @param config The configuration to hydrate
     */
    public void hydrateComponents(Configuration config);
    
    /**
     * Hydrates all the <code>AcsService</code> objects associated with a given configuration.
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated acs services.
     * 
     * @param config The configuration to hydrate
     */
    public void hydrateAcsServices(Configuration config);

    /**
     * Hydrates all the <code>Container</code> objects associated with a <code>Configuration</code> object
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated containers
     * 
     * @param config The given configuration
     */
    public void hydrateContainers(Configuration config);

    /**
     * Hydrates all the <code>Computer</code> objects associated with a <code>Configuration</code> object
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated computers
     * 
     * @param config The given configuration
     */
    public void hydrateComputers(Configuration config);
    
    /**
     * Hydrates all the <code>BACIProperty</code> objects associated with a <code>Configuration</code> object
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated baci properties.
     * 
     * @param config The given configuration
     */
    public void hydrateBACIProperties(Configuration config);

    /**
     * Hydrates all the <code>EventChannel</code> objects associated with a <code>Configuration</code> object
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated event channels
     * 
     * @param config The given configuration
     */
    public void hydrateEventChannels(Configuration config);
    
    /**
     * Hydrates all the <code>NetworkDevice</code> objects associated with a <code>Configuration</code> object
     * There is no return value. Instead, the given configuration now contains the actual
     * contents of its associated network devices.
     * 
     * @param config The given configuration
     */
    public void hydrateNetworkDevices(Configuration config);

	/**
	 * Hydrates the Schema objects for a given configuration
	 * @param config The configuration
	 */
	public void hydrateSchemas(Configuration config);
	
    /**
     * Hydrates the configuration's alarm definitions.
     * @param domainObject the configuration for which to hydrate the alarm definitions.
     * @throws ServiceException if there is a problem hydrating.
     */
     public void hydrateAlarmDefinitions(Configuration domainObject) throws ServiceException;
     
     /**
      * Hydrates the configuration's reduction thresholds.
      * @param domainObject the configuration for which to hydrate the reduction thresholds.
      * @throws ServiceException if there is a problem hydrating.
      */
      public void hydrateReductionThresholds(Configuration domainObject) throws ServiceException;
     
    /**
     * Hydrates the configuration's alarm categories.
     * @param domainObject the configuration for which to hydrate the alarm categories.
     * @throws ServiceException if there is a problem hydrating.
     */
     public void hydrateAlarmCategories(Configuration domainObject) throws ServiceException;
     
     /**
      * Hydrates the configuration's reduction links.
      * @param domainObject the configuration for which to hydrate the alarm categories.
      * @throws ServiceException if there is a problem hydrating.
      */
      public void hydrateReductionLinks(Configuration domainObject) throws ServiceException;
      
     /**
      * Hydrates the configuration's fault families.
      * @param domainObject the configuration for which to hydrate the fault families.
      * @throws ServiceException if there is a problem hydrating.
      */
      public void hydrateFaultFamilies(Configuration domainObject) throws ServiceException;
      
      /**
       * Hydrates all the <code>NotificationServiceMapping</code> objects associated with a given configuration.
       * There is no return value. Instead, the given configuration now contains the actual
       * contents of its associated NotificationServiceMapping objects.
       * 
       * @param config The configuration to hydrate
       */
      public void hydrateNotificationServiceMappings(Configuration config);

      /**
       * Hydrates all the <code>Manager</code> objects associated with a given configuration.
       * There is no return value. Instead, the given configuration now contains the actual
       * contents of its associated Manager objects.
       * 
       * @param config The configuration to hydrate
       */
      public void hydrateManagers(Configuration config);
}
