/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.List;
import java.util.Set;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

public class StartupScenarioServiceImpl extends TmcdbAbstractService implements	StartupScenarioService 
{
	/**
     * @see alma.obops.dam.service.IConfigurationService#findAll()
     */
    @SuppressWarnings("unchecked")
    public List<StartupScenario> findAll() throws ServiceException {
 	   List<StartupScenario> startups = (List<StartupScenario>)getDao().findAll(getDomainClass());
       return startups;
    }
    
	@Override
	public Class<?> getDomainClass() {
		return StartupScenario.class;
	}
	
	@Override
	public BaseElementStartup addBaseElementToStartupScenario( BaseElement baseElement, StartupScenario scenario ) throws ServiceException 
	{
        // Look for that BaseElement, do we have it already?
        for( BaseElementStartup bes : scenario.getBaseElementStartups() )
        {
            BaseElement be = bes.getBaseElement();
            if( null != be && be.equals( baseElement )) {
                return null;   // YES, we do have it, nothing to do
            }
        }

        // NO, we don't have it and can add it
        BaseElementStartup bes = new BaseElementStartup( baseElement, scenario );
        bes.setSimulated(false);
        this.getDao().saveOrUpdate(scenario);
        return bes;
	}
	
	@Override
	public void hydrateBaseElementStartups(Object domainObject)
	{
		StartupScenario startup = (StartupScenario)domainObject;
		this.getDao().reAttach(startup);
		startup.getBaseElementStartups().size();
		hydrateSet(startup.getBaseElementStartups());
	}

	private void hydrateSet(Set<BaseElementStartup> setToHydrate)
	{
		setToHydrate.size();
		for(BaseElementStartup bes : setToHydrate) {
			bes.getAssemblyStartups().size();
			if(null != bes.getChildren()) {
				hydrateSet(bes.getChildren());
			}
		}	
	}
	
	@Override
	public void hydrateAssemblyStartups(Object domainObject)
	{
		StartupScenario startup = (StartupScenario)domainObject;
		this.getDao().reAttach(startup);
		startup.getAssemblyStartups().size();
		hydrateSet(startup.getBaseElementStartups());
	}
	
	@Override
	public StartupScenario cloneStartupScenario(StartupScenario startup, String clonedName) 
	{
		// clone the startup
		StartupScenario clonedStartup = this.getDao().cloneStartupScenario(startup, clonedName);

		// return the clone
		return clonedStartup;
	}

	@SuppressWarnings("unchecked")
	@Override
	public StartupScenario findByNameWithinConfiguration(HwConfiguration config, String name) 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(StartupScenario.class).
		add(Restrictions.eq("name", name)).
		createAlias("configuration", "configAlias").
		add(Restrictions.eq("configAlias.id", config.getId()));
		
		List<StartupScenario> results = (List<StartupScenario>)this.getDao().find(searchCriteria);

		StartupScenario retVal = null;
		if(results != null && results.size() == 1) 
		{
			retVal = results.get(0);
		} 
		else if(results != null && results.size() > 1) {
			throw new IllegalStateException("Database constraints have been violated for uniqueness of StartupScenario name within configuration!");
		}

		return retVal;
	}
}
