/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.List;

import alma.obops.dam.Service;
import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.BaseElement;
import alma.tmcdb.domain.BaseElementStartup;
import alma.tmcdb.domain.HwConfiguration;
import alma.tmcdb.domain.StartupScenario;

public interface StartupScenarioService extends Service 
{
    /**
     * @return all the startup scenarios
     * @throws ServiceException
     */
    public List<StartupScenario> findAll() throws ServiceException;

    /**
     * hydrates the base element startups of the configuration.
     * @param domainObject the configuration to hydrate.
     * @throws ServiceException if there is a problem
     */
    public void hydrateBaseElementStartups(Object domainObject);

    /**
     * hydrates the assembly startups of the configuration.
     * @param domainObject the configuration to hydrate.
     * @throws ServiceException if there is a problem
     */
    public void hydrateAssemblyStartups(Object domainObject);

    /**
     * Clones an existing startup scenario within a configuration.
     * @param scenarioToClone the startup scenario to clone.
     * @param clonedName the name of the newly cloned startup scenario; must be unique within a configuration.
     * @return the new cloned (and persisted) startup scenario instance
     */
    public StartupScenario cloneStartupScenario(StartupScenario scenarioToClone, String clonedName);
    
    /**
     * Finds a startup scenario by name, within the given configuration.
     * @param config the configuration in which to search.
     * @param name the name of the startup scenario for which to search.
     * @return the startup scenario matching the given name, within the given configuration, or null if none;
     *         note that there can be only zero or one match, as there is a uniqueness constraint 
     *         in the database (on the configuration's id and the name).
     */
    public StartupScenario findByNameWithinConfiguration(HwConfiguration config, String name);
    
    /**
     * Add the input BaseElement to the input StartupScenario (by way of a new
     * BaseElementStartup instance), after verifying that the BaseElement is not
     * already present. If it is, nothing happens.
     * 
     * @return <code>null</code> if the BaseElement was already present, and no
     *         adding was performed; the new BaseElementStartup otherwise.
     * 
     * @throws ServiceException
     *             If anything goes wrong
     */
    public BaseElementStartup addBaseElementToStartupScenario( BaseElement baseElement, StartupScenario scenario ) throws ServiceException;
}
