/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
/**
 * LruTypeServiceImpl.java
 *
 * Copyright European Southern Observatory 2008
 */

package alma.obops.dam.tmcdb.service;

import java.util.List;

import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.obops.dam.HibernateDao;
import alma.obops.dam.ServiceException;
import alma.tmcdb.domain.AssemblyRole;
import alma.tmcdb.domain.AssemblyType;
import alma.tmcdb.domain.BaseElementStartupType;
import alma.tmcdb.domain.BaseElementType;
import alma.tmcdb.domain.LruType;

/**
 * Business layer for LruType
 * 
 * @author rkurowsk, Dec 12, 2008
 * 
 */



public class LruTypeServiceImpl extends TmcdbAbstractService implements
		LruTypeService {

	/**
	 * Public constructor
	 */
	public LruTypeServiceImpl() {
		super();
	}

	/**
	 * @see alma.obops.dam.Service#getDomainClass()
	 */
	@Override
	public Class<?> getDomainClass() {
		return LruType.class;
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.LruTypeService#findByBaseElementStartupType(alma.tmcdb.domain.BaseElementStartupType)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public LruType[] findByBaseElementStartupType(BaseElementStartupType baseElementStartupType) throws ServiceException 
	{	
		BaseElementType baseElementType = BaseElementType.Antenna;
		switch(baseElementStartupType) {
		case Antenna:
			baseElementType = BaseElementType.Antenna;
			break;
		case Array:
			baseElementType = BaseElementType.Array;
			break;
		case Pad:
			baseElementType = BaseElementType.Pad;
			break;
		case CentralLO:
			baseElementType = BaseElementType.CentralLO;
			break;
		case AOSTiming:
			baseElementType = BaseElementType.AOSTiming;
			break;
		case FrontEnd:
			baseElementType = BaseElementType.FrontEnd;
			break;
		case WeatherStationController:
			baseElementType = BaseElementType.WeatherStationController;
			break;
		case HolographyTower:
			baseElementType = BaseElementType.HolographyTower;
			break;
		case PhotonicReference1:
		case PhotonicReference2:
		case PhotonicReference3:
		case PhotonicReference4:
		case PhotonicReference5:
		case PhotonicReference6:
			baseElementType = BaseElementType.PhotonicReference;
			break;
		}
		
		// create the criteria
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(LruType.class)
		.setResultTransformer(DetachedCriteria.DISTINCT_ROOT_ENTITY)
		.createCriteria("assemblyTypes", DetachedCriteria.LEFT_JOIN)
		.add(Restrictions.eq("baseElementType", baseElementType));
 
		// perform the query
		List<LruType> results = (List<LruType>) this.getDao().find(searchCriteria);
		
        return results.toArray(new LruType[0]);		
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.LruTypeService#findAllAssemblyRoles()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public AssemblyRole[] findAllAssemblyRoles() throws ServiceException {

		List<AssemblyRole> assemblyRoles = (List<AssemblyRole>)getDao().findAll(AssemblyRole.class);
		return assemblyRoles.toArray(new AssemblyRole[0]);
		
	}

	/* (non-Javadoc)
	 * @see alma.obops.dam.tmcdb.service.LruTypeService#findAllAssemblyTypes()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public AssemblyType[] findAllAssemblyTypes() throws ServiceException {
		List<AssemblyType> assemblyTypes = (List<AssemblyType>)getDao().findAll(AssemblyType.class);
		return assemblyTypes.toArray(new AssemblyType[0]);
	}

	/**
	 * Read an instance from the database; this is a thin wrapper around
	 * {@link HibernateDao#read(String)}.
	 * 
	 * @return The LruType whose name is given as input
	 * @throws ServiceException
	 */
	public LruType read(String lruName) throws ServiceException {

		return (LruType) getDao().read(lruName, LruType.class);

	}
	
	@Override
	public void hydrateToAssemblyTypes(Object domainObject)
	{
		LruType lruType = (LruType) domainObject;
		this.getDao().reAttach(lruType);
		lruType.getAssemblyTypes().size();
		for(AssemblyType assemblyType : lruType.getAssemblyTypes()) {
			assemblyType.getRoles().size();
			assemblyType.getComponentType();
		}
	}
	
	/**
     * @see alma.obops.dam.service.IConfigurationService#findAll()
     */
    @SuppressWarnings("unchecked")
    public List<LruType> findAll() throws ServiceException {
 	   List<LruType> lrutypes = (List<LruType>)getDao().findAll(getDomainClass());
       return lrutypes;
    }
    
}
