/*******************************************************************************
 * ALMA - Atacama Large Millimeter Array
 * Copyright (c) ESO - European Southern Observatory, 2011
 * (in the framework of the ALMA collaboration).
 * All rights reserved.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *******************************************************************************/
package alma.obops.dam.tmcdb.service;

import java.util.List;

import org.hibernate.Hibernate;
import org.hibernate.criterion.CriteriaSpecification;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Restrictions;

import alma.acs.tmcdb.Configuration;
import alma.acs.tmcdb.FaultFamily;
import alma.acs.tmcdb.FaultMember;
import alma.obops.dam.ServiceException;

/**
 * Implementation of the fault member service interface.
 * @author sharring
 */
public class FaultMemberServiceImpl extends TmcdbAbstractService implements
		FaultMemberService {

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultMember> findAll() throws ServiceException 
	{
	    List<FaultMember> faultMembers = (List<FaultMember>)this.getDao().findAll(getDomainClass());
		return faultMembers;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<FaultMember> findAllInConfiguration(Configuration config) throws ServiceException 
	{
		DetachedCriteria searchCriteria = DetachedCriteria.forClass(FaultMember.class).
		   createAlias("faultFamily", "ff").
		   createAlias("ff.configuration", "con").
		   add(Restrictions.eq("con.configurationId", config.getConfigurationId()));

		List<FaultMember> results = (List<FaultMember>)this.getDao().find(searchCriteria);

		return results;
	}

	@Override
	public Class<?> getDomainClass() {
		return FaultMember.class;
	}

	@Override 
	public void hydrate(Object domainObject)
	{
		if(domainObject instanceof FaultMember) {
			FaultMember faultMember = (FaultMember) domainObject;
			this.getDao().reAttach(faultMember);
			if(null != faultMember.getLocation()) {
				faultMember.getLocation().getMnemonic();
			}
		}
	}

	@Override
	public FaultMember hydrateAndMerge(FaultMember faultMember) 
	{
		faultMember = (FaultMember) this.getDao().merge(faultMember);
		this.getDao().reAttach(faultMember);
		if(null != faultMember.getLocation()) {
			faultMember.getLocation().getMnemonic();
		}
		return faultMember;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public List<FaultMember> findFaultMembersByRegExInConfig(String fmregex, String ffregex, Configuration config) 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultMember.class);
		
		baseCriteria.add(
				Restrictions.sqlRestriction("regexp_like({alias}.memberName, ?)", fmregex, Hibernate.STRING)
		);
		
		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);
		
		subCriteria.createAlias("ff.configuration", "con").add
		(
				Restrictions.and
				(
						Restrictions.eq("con.configurationId", config.getConfigurationId()), 
						Restrictions.sqlRestriction("regexp_like({alias}.familyName, ?)", ffregex, Hibernate.STRING)
				)
		);

		List<FaultMember> fms  = (List<FaultMember>)this.getDao().find(baseCriteria);
		
		return fms;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<FaultMember> findByFaultFamily(FaultFamily ff)
			throws ServiceException 
	{
		DetachedCriteria baseCriteria = DetachedCriteria.forClass(FaultMember.class);
		baseCriteria.add(Restrictions.eq("faultFamily", ff));
		DetachedCriteria subCriteria = baseCriteria.createCriteria("faultFamily", "ff", CriteriaSpecification.LEFT_JOIN);

		subCriteria.createAlias("ff.configuration", "con").add
		(
				Restrictions.eq("con.configurationId", ff.getConfiguration().getConfigurationId()) 
		);

		List<FaultMember> fms  = (List<FaultMember>)this.getDao().find(baseCriteria);

		return fms;
	}
}
